/**
 * Health Check API Route
 * GET /api/health
 * Returns system health status for monitoring
 */

import { NextRequest } from 'next/server';
import { HealthChecker, logger, performanceMonitor } from '@/lib/error-handler';
import { createSecureResponse } from '@/lib/auth';

export async function GET(request: NextRequest) {
  const startTime = Date.now();

  try {
    // Get health status
    const healthStatus = await HealthChecker.getHealthStatus();

    // Add performance metrics
    const uptime = process.uptime();
    const memoryUsage = process.memoryUsage();

    const response = {
      ...healthStatus,
      uptime,
      memory: {
        rss: memoryUsage.rss,
        heapTotal: memoryUsage.heapTotal,
        heapUsed: memoryUsage.heapUsed,
        external: memoryUsage.external,
      },
      version: process.env.npm_package_version || '1.0.0',
      environment: process.env.NODE_ENV || 'development',
    };

    // Log health check
    logger.debug('Health check performed', {
      status: healthStatus.status,
      responseTime: Date.now() - startTime,
    });

    // Record performance
    performanceMonitor.recordMetric('health_check', Date.now() - startTime);

    // Return appropriate status code based on health
    const statusCode = healthStatus.status === 'healthy' ? 200 :
      healthStatus.status === 'degraded' ? 200 : 503;

    return createSecureResponse(response, statusCode);

  } catch (error) {
    logger.error('Health check failed', { error: (error as Error).message });

    return createSecureResponse(
      {
        status: 'unhealthy',
        error: 'Health check failed',
        timestamp: new Date().toISOString(),
      },
      503
    );
  }
}
