/**
 * Feedback Submission API Route
 * POST /api/feedback
 * Allows users to submit feedback, bug reports, and feature requests
 */

import { NextRequest } from 'next/server';
import { z } from 'zod';
import { AuthMiddleware, createSecureResponse, DatabaseService } from '@/lib/auth';
import { logger } from '@/lib/error-handler';
import nodemailer from 'nodemailer';

const feedbackSchema = z.object({
  type: z.enum(['bug', 'feature', 'improvement', 'question', 'other']),
  title: z.string().min(1, 'Title is required').max(100, 'Title is too long'),
  description: z.string().min(10, 'Description must be at least 10 characters').max(1000, 'Description is too long'),
});

export async function POST(request: NextRequest) {
  try {
    // Require authentication
    const authResponse = await AuthMiddleware.requireAuth(request);
    if (authResponse) return authResponse;

    const user = (request as any).user;

    // Parse request body
    const body = await request.json();

    // Validate input
    const { type, title, description } = feedbackSchema.parse(body);

    const priority =
      type === 'bug' ? 'HIGH' :
        type === 'feature' ? 'MEDIUM' :
          type === 'improvement' ? 'MEDIUM' :
            type === 'question' ? 'LOW' :
              'LOW';

    // Persist feedback in database
    const created = await DatabaseService.createFeedback({
      userId: user.id,
      feedbackType: type,
      subject: title,
      message: description,
      priority,
      status: 'OPEN',
      metadata: {
      userAgent: request.headers.get('user-agent') || undefined,
        ip: (request as any).ip,
      },
    });

    logger.info('Feedback submitted', {
      feedbackId: created.id,
      userId: user.id,
      type,
      title,
    });

    // Send email notification to admin
    try {
      const transporter = nodemailer.createTransport({
        host: process.env.SMTP_HOST || 'localhost',
        port: parseInt(process.env.SMTP_PORT || '25'),
        secure: false,
        tls: { rejectUnauthorized: false },
      });

      const priorityEmoji = priority === 'HIGH' ? '🔴' : priority === 'MEDIUM' ? '🟡' : '🟢';
      const typeEmoji = type === 'bug' ? '🐛' : type === 'feature' ? '✨' : type === 'improvement' ? '📈' : type === 'question' ? '❓' : '📝';

      await transporter.sendMail({
        from: `"${process.env.FROM_NAME || 'EventheOdds.ai'}" <${process.env.EMAIL_FROM || 'noreply@eventheodds.ai'}>`,
        to: process.env.SUPPORT_EMAIL || 'support@eventheodds.ai',
        subject: `${priorityEmoji} ${typeEmoji} New ${type.toUpperCase()}: ${title}`,
        html: `
          <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
            <h2 style="color: #667eea;">New Feedback Received</h2>
            <table style="width: 100%; border-collapse: collapse;">
              <tr><td style="padding: 8px; border-bottom: 1px solid #ddd;"><strong>Type:</strong></td><td style="padding: 8px; border-bottom: 1px solid #ddd;">${typeEmoji} ${type}</td></tr>
              <tr><td style="padding: 8px; border-bottom: 1px solid #ddd;"><strong>Priority:</strong></td><td style="padding: 8px; border-bottom: 1px solid #ddd;">${priorityEmoji} ${priority}</td></tr>
              <tr><td style="padding: 8px; border-bottom: 1px solid #ddd;"><strong>From:</strong></td><td style="padding: 8px; border-bottom: 1px solid #ddd;">${user.email}</td></tr>
              <tr><td style="padding: 8px; border-bottom: 1px solid #ddd;"><strong>User ID:</strong></td><td style="padding: 8px; border-bottom: 1px solid #ddd;">${user.id}</td></tr>
              <tr><td style="padding: 8px; border-bottom: 1px solid #ddd;"><strong>Title:</strong></td><td style="padding: 8px; border-bottom: 1px solid #ddd;">${title}</td></tr>
            </table>
            <h3 style="margin-top: 20px;">Description:</h3>
            <div style="background: #f8f9fa; padding: 15px; border-radius: 6px; white-space: pre-wrap;">${description}</div>
            <p style="margin-top: 20px; color: #666; font-size: 12px;">Feedback ID: ${created.id}</p>
          </div>
        `,
        text: `New ${type} feedback from ${user.email}\n\nTitle: ${title}\nPriority: ${priority}\n\n${description}\n\nFeedback ID: ${created.id}`,
      });

      logger.info('Feedback notification email sent', { feedbackId: created.id });
    } catch (emailError) {
      console.warn('Failed to send feedback notification email:', emailError);
      // Don't fail the request if email fails
    }

    // Record analytics event
    try {
      await DatabaseService.recordUserEvent(user.id, 'FEEDBACK_SUBMITTED', {
        feedbackId: created.id,
        type,
        title,
        priority,
      });
    } catch (e) {
      console.warn('Failed to record FEEDBACK_SUBMITTED', e);
    }

    return createSecureResponse({
      success: true,
      message: 'Thank you for your feedback! We\'ll review it and get back to you soon.',
      feedbackId: created.id,
      estimatedResponse: type === 'bug' ? '24-48 hours' : type === 'feature' ? '1-2 weeks' : 'Same day'
    }, 201, user);

  } catch (error) {
    console.error('Feedback submission error:', error);

    if (error instanceof z.ZodError) {
      return createSecureResponse(
        { error: 'Validation failed', details: error.issues },
        400
      );
    }

    return createSecureResponse(
      { error: 'Failed to submit feedback. Please try again.' },
      500
    );
  }
}

