/**
 * Error Reporting API Route
 * POST /api/errors/report
 * Accepts error reports from client-side and logs them
 */

import { NextRequest } from 'next/server';
import { z } from 'zod';
import { AuthMiddleware, SecurityMiddleware, createSecureResponse } from '@/lib/auth';
import { ErrorHandler, logger } from '@/lib/error-handler';

const errorReportSchema = z.object({
  type: z.string(),
  message: z.string(),
  stack: z.string().optional(),
  componentStack: z.string().optional(),
  url: z.string().optional(),
  userAgent: z.string().optional(),
  timestamp: z.string().optional(),
  metadata: z.record(z.string(), z.any()).optional(),
});

export async function POST(request: NextRequest) {
  try {
    // Optional authentication (allow anonymous error reports)
    await AuthMiddleware.optionalAuth(request);
    const user = (request as any).user;

    // Rate limiting (allow higher limit for error reports)
    const rateLimitResponse = SecurityMiddleware.rateLimit(request, 20, 60 * 1000); // 20 reports per minute
    if (rateLimitResponse) return rateLimitResponse;

    // Parse and validate request
    const json = await request.json();
    const validation = SecurityMiddleware.validateInput(json, {
      required: ['type', 'message'],
      stringFields: ['type', 'message', 'stack', 'componentStack', 'url', 'userAgent', 'timestamp'],
      maxLength: {
        type: 50,
        message: 1000,
        stack: 5000,
        componentStack: 5000,
        url: 2000,
        userAgent: 500,
      }
    });

    if (!validation.valid) {
      return createSecureResponse(
        { error: 'Validation failed', details: validation.errors },
        400,
        user
      );
    }

    const errorData = errorReportSchema.parse(json);

    // Log the client-side error
    logger.error('Client-side error reported', {
      type: errorData.type,
      message: errorData.message,
      stack: errorData.stack,
      componentStack: errorData.componentStack,
      url: errorData.url,
      userAgent: errorData.userAgent,
      userId: user?.id,
      timestamp: errorData.timestamp || new Date().toISOString(),
      metadata: errorData.metadata,
    });

    // Store in database for analysis
    try {
      await ErrorHandler.handleError(new Error(`Client Error: ${errorData.message}`), {
        userId: user?.id,
        url: errorData.url,
        userAgent: errorData.userAgent,
        metadata: {
          errorType: errorData.type,
          componentStack: errorData.componentStack,
          clientTimestamp: errorData.timestamp,
          ...errorData.metadata,
        },
      }
      );
    } catch (dbError) {
      logger.error('Failed to store client error in database', {
        originalError: errorData,
        dbError: (dbError as Error).message,
      });
    }

    return createSecureResponse(
      { success: true, message: 'Error report received' },
      200,
      user
    );

  } catch (error) {
    logger.error('Error in error reporting endpoint', { error: (error as Error).message });

    // Don't return error details to avoid infinite loops
    return createSecureResponse(
      { error: 'Failed to process error report' },
      500
    );
  }
}
