import { NextRequest } from 'next/server';
import { z } from 'zod';

import { callGrok4Fast } from '@/lib/chat';
import { AuthMiddleware, SecurityMiddleware, createSecureResponse, DatabaseService, checkFeatureAccess } from '@/lib/auth';
import { PerformanceMiddleware, CacheInvalidation } from '@/lib/performance-middleware';

const chatRequestSchema = z.object({
  message: z.string().min(1, 'Message cannot be empty').max(2000, 'Message too long'),
  domain: z.enum(['sports', 'crypto', 'stocks', 'forex']),
  conversationId: z.string().optional(),
});

// Time period validation function
function validateTimePeriodRequirement(message: string, domain: string): { isValid: boolean; message?: string } {
  const messageLower = message.toLowerCase();

  // Only require a time period for requests that actually run a backtest/simulation.
  const isBacktestRequest =
    messageLower.includes('backtest') ||
    messageLower.includes('test strategy') ||
    messageLower.includes('run simulation') ||
    messageLower.includes('simulate') ||
    messageLower.includes('analyze performance') ||
    messageLower.includes('parlay') ||
    messageLower.includes('round robin') ||
    messageLower.includes('optimize') ||
    messageLower.includes('optimise') ||
    messageLower.includes('parameter');

  if (!isBacktestRequest) {
    return { isValid: true };
  }

  // Check for time period indicators
  const timeIndicators = [
    // Years: 2023, 2024, 2024-25, etc.
    /\b20\d{2}(-20\d{2})?\b/,
    // Months: january, february, etc.
    /\b(january|february|march|april|may|june|july|august|september|october|november|december)\b/i,
    // Relative time: last 6 months, past year, etc.
    /\b(last|past|previous)\s+\d*\s*(month|year|week|day)s?\b/i,
    // Date ranges: 2023-2024, jan-jun, etc.
    /\b\d{4}-\d{4}\b/,
    // Seasons for sports: 2023-24 season, regular season, etc.
    /\b(season|regular season|playoffs?|play-off)\b/i
  ];

  const hasTimePeriod = timeIndicators.some(pattern => pattern.test(messageLower));

  if (!hasTimePeriod) {
    let example = "";
    if (domain === 'sports') {
      example = "Please specify the time period for backtesting (e.g., '2024-25 NBA season', '2024 regular season').";
    } else {
      example = "Please specify the time period for backtesting (e.g., 'January 2024', 'last 6 months', '2024-2025').";
    }

    return {
      isValid: false,
      message: `⚠️ **Time Period Required**\n\nTo ensure accurate backtesting results, you must specify a time period for your strategy testing.\n\n${example}\n\nThis helps avoid overfitting and provides transparency about when your strategy was tested.`
    };
  }

  return { isValid: true };
}

// Subscription tier checking
async function checkUserSubscriptionTier(request: NextRequest): Promise<string> {
  try {
    // Extract user from JWT token (simplified - in real app this would validate the token)
    const authHeader = request.headers.get('authorization');
    if (!authHeader) return 'free';

    // For now, return 'free' - in production this would check the actual subscription
    // This is a placeholder for the subscription checking logic
    // Check actual subscription from database
    const userProfile = await DatabaseService.findUserById((request as any).user?.id);
    return userProfile?.subscriptionTier?.toLowerCase() || 'free';

  } catch (error) {
    console.error('Error checking subscription tier:', error);
    return 'free';
  }
}

// Helper functions for visualization generation
// PRODUCTION NOTE: This generates SAMPLE data for demonstration purposes only
// Real backtests should use actual historical odds + results data (BallDontLie + our local datasets)
function generateSampleTradesForDemo(domain: string): { trades: any[]; isSampleData: boolean } {
  // Only generate sample data if explicitly requested or in demo mode
  const isDemo = process.env.DEMO_MODE === 'true';

  if (!isDemo && process.env.NODE_ENV === 'production') {
    // In production without demo mode, return empty - require real data
    console.warn('[Visualization] Skipping sample data generation in production');
    return { trades: [], isSampleData: false };
  }

  // Generate sample trade data with clear labeling
  const numTrades = 15;
  const trades = [];
  let currentDate = new Date();
  currentDate.setDate(currentDate.getDate() - numTrades);

  for (let i = 0; i < numTrades; i++) {
    const outcome = Math.random() > 0.45 ? 'win' : 'loss'; // 55% win rate sample
    const profit = outcome === 'win'
      ? (Math.random() * 150 + 30) // $30-180 profit
      : -(Math.random() * 120 + 20); // -$20-140 loss

    trades.push({
      id: `sample_trade_${i}`,
      date: currentDate.toISOString().split('T')[0],
      outcome,
      profit: Number(profit.toFixed(2)),
      label: `[SAMPLE] ${domain.charAt(0).toUpperCase() + domain.slice(1)} Trade ${i + 1}`,
      isSampleData: true, // Flag to indicate this is sample data
      context: { note: 'This is sample data for demonstration purposes only' }
    });

    currentDate.setDate(currentDate.getDate() + 1);
  }

  return { trades, isSampleData: true };
}

async function generateBacktestVisualization(backtestResults: any): Promise<any> {
  if (!backtestResults.success || !backtestResults.trades) {
    return null;
  }

  try {
    // Generate profit chart
    const profitChart = await fetch(`${process.env.NEXT_PUBLIC_BASE_URL || 'http://localhost:3000'}/api/visualization/generate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        chartType: 'profit_chart',
        trades: backtestResults.trades,
        market: backtestResults.market || 'general',
        chartFormat: 'plotly'
      })
    });

    if (profitChart.ok) {
      const chartResult = await profitChart.json();
      return chartResult.data;
    }

    console.error('Failed to generate backtest visualization');
    return null;
  } catch (error) {
    console.error('Error generating backtest visualization:', error);
    return null;
  }
}

export async function POST(request: NextRequest) {
  return PerformanceMiddleware.trackPerformance(request, async () => {
    try {
      // Require authentication
      const authResponse = await AuthMiddleware.requireAuth(request);
      if (authResponse) return authResponse;

      const user = (request as any).user;

      // Rate limiting for chat requests
      const rateLimitResponse = SecurityMiddleware.rateLimit(request, 50, 60 * 60 * 1000); // 50 messages per hour
      if (rateLimitResponse) return rateLimitResponse;

      // Parse and validate request
      const json = await request.json();
      const validation = SecurityMiddleware.validateInput(json, {
        required: ['message', 'domain'],
        stringFields: ['message', 'domain', 'conversationId'],
        maxLength: { message: 2000 }
      });

      if (!validation.valid) {
        return createSecureResponse(
          { error: 'Validation failed', details: validation.errors },
          400,
          user
        );
      }

      const { message, domain, conversationId } = chatRequestSchema.parse(json);

      // Validate time period requirement
      const timePeriodValidation = validateTimePeriodRequirement(message, domain);
      if (!timePeriodValidation.isValid) {
        return createSecureResponse(
          {
            assistantContent: timePeriodValidation.message,
            timePeriodRequired: true,
            requiresTimePeriod: true
          },
          200,
          user
        );
      }

      // Sanitize user input
      const sanitizedMessage = SecurityMiddleware.sanitizeInput(message);

      // Check subscription limits
      const userProfile = await DatabaseService.findUserById(user.id);
      if (!userProfile) {
        return createSecureResponse({ error: 'User not found' }, 404, user);
      }

      // FREE tier users have full Pro access (they're on a trial subscription)
      const effectiveTier = userProfile.subscriptionTier === 'FREE' ? 'PRO' : userProfile.subscriptionTier;
      const isTrialActive = userProfile.subscriptionTier === 'FREE' || userProfile.subscriptionTier === 'TRIAL';
      const userStats = await DatabaseService.getUserStats(user.id);
      const maxConversationsFeature = checkFeatureAccess(effectiveTier, 'maxConversations');
      const maxConversations = typeof maxConversationsFeature === 'number' ? maxConversationsFeature : (maxConversationsFeature ? -1 : 10);

      if (maxConversations !== -1 && userStats.totalConversations >= maxConversations) {
        const tierName = isTrialActive ? 'trial' : userProfile.subscriptionTier.toLowerCase();
        return createSecureResponse(
          {
            error: 'Conversation limit reached',
            message: `Your ${tierName} plan allows ${maxConversations} conversations. ${isTrialActive ? 'Upgrade to Pro to continue chatting.' : 'Upgrade to continue.'}`,
            currentUsage: userStats.totalConversations,
            limit: maxConversations,
            isTrialActive,
            trialEndsAt: userProfile.trialEndsAt
          },
          403,
          user
        );
      }

      // Find or create conversation
      let conversation;
      if (conversationId) {
        conversation = await DatabaseService.findConversationById(conversationId);
        if (!conversation || conversation.userId !== user.id) {
          return createSecureResponse(
            { error: 'Conversation not found or access denied' },
            404,
            user
          );
        }
      } else {
        // Create new conversation
        conversation = await DatabaseService.createConversation({
          title: sanitizedMessage.substring(0, 50) + (sanitizedMessage.length > 50 ? '...' : ''),
          domain,
          userId: user.id,
        });
      }

      // Add user message to conversation
      await DatabaseService.addMessageToConversation(
        conversation.id,
        'USER',
        sanitizedMessage
      );

      // Check for premium features (parameter optimization)
      const isOptimizationRequest = sanitizedMessage.toLowerCase().includes('optimize') ||
        sanitizedMessage.toLowerCase().includes('optimise') ||
        sanitizedMessage.toLowerCase().includes('parameter');

      if (isOptimizationRequest) {
        // Check subscription tier for optimization access
        if (effectiveTier === 'FREE') {
          const errorResponse = {
            error: 'Parameter optimization requires a paid subscription',
            message: '❌ **Parameter optimization requires a paid subscription.**\n\n**Subscription Options:**\n• **Pro ($20/month)**: Limited parameter optimization (up to 20 evaluations)\n• **Elite ($200/month)**: Full parameter optimization with advanced algorithms\n\n**Upgrade your subscription to unlock parameter optimization!**\n\nYou can still use basic backtesting and strategy generation on the FREE tier.',
            subscriptionRequired: true,
            feature: 'parameter_optimization'
          };
          return createSecureResponse(errorResponse, 403, user);
        }

        // For Pro tier, limit evaluations
        if (effectiveTier === 'PRO' && sanitizedMessage.toLowerCase().includes('evaluations')) {
          // The limitation will be enforced in the chat.ts logic
          console.log('Pro tier user requesting optimization - limiting evaluations to 20');
        }
      }

      // Generate AI response
      const aiResponse = await callGrok4Fast(sanitizedMessage, domain, user.id);

      // Record AI generation usage (monthly usage counter)
      try {
        await DatabaseService.recordUserEvent(user.id, 'AI_GENERATION', {
          domain,
          conversationId: conversation.id,
          messageLength: sanitizedMessage.length,
          hasRagInsight: !!aiResponse.ragInsight,
          hasBacktest: !!aiResponse.backtestResults,
        });
      } catch (e) {
        console.warn('Failed to record AI_GENERATION analytics', e);
      }

      // Generate visualization if requested in the message
      let visualization = null;
      let visualizationIsSampleData = false;
      if (sanitizedMessage.toLowerCase().includes('chart') ||
        sanitizedMessage.toLowerCase().includes('graph') ||
        sanitizedMessage.toLowerCase().includes('visual') ||
        sanitizedMessage.toLowerCase().includes('plot') ||
        aiResponse.message.content.toLowerCase().includes('chart') ||
        aiResponse.message.content.toLowerCase().includes('graph')) {

        // Use real backtest data if available, otherwise generate sample data with disclaimer
        if (aiResponse.backtestResults?.trades?.length > 0) {
          // Real backtest data exists - use it
          try {
            visualization = await generateBacktestVisualization({
              success: true,
              trades: aiResponse.backtestResults.trades
            });
          } catch (error) {
            console.error('Visualization generation failed:', error);
          }
        } else {
          // No real data - generate sample data for demonstration only
          const sampleResult = generateSampleTradesForDemo(domain);
          if (sampleResult.trades.length > 0) {
            try {
              visualization = await generateBacktestVisualization({
                success: true,
                trades: sampleResult.trades
              });
              visualizationIsSampleData = sampleResult.isSampleData;
            } catch (error) {
              console.error('Sample visualization generation failed:', error);
            }
          }
        }
      }

      // Add AI response to conversation (store the actual assistant message, not just extracted code)
      if (aiResponse?.message?.content) {
        await DatabaseService.addMessageToConversation(
          conversation.id,
          'ASSISTANT',
          aiResponse.message.content,
          (aiResponse as any).tokenCount
        );
      }

      // Record analytics
      await DatabaseService.recordUserEvent(user.id, 'CHAT_MESSAGE_SENT', {
        domain,
        conversationId: conversation.id,
        messageLength: sanitizedMessage.length,
        hasStrategyCode: !!aiResponse.strategyCode
      });

      // Persist successful backtests and record usage
      if (aiResponse.backtestResults && aiResponse.backtestResults.success) {
        try {
          const bt = aiResponse.backtestResults;
          const market = (bt.market || bt.sport || domain || 'sports').toString();
          const strategy = (aiResponse.strategyName || bt.strategy_name || bt.strategyName || 'AI Strategy').toString();

          const stored = await DatabaseService.createBacktestResult({
            userId: user.id,
            market,
            strategyName: strategy,
            strategyCode: aiResponse.strategyCode || null,
            parameters: {
              prompt: sanitizedMessage,
              domain,
              conversationId: conversation.id,
            },
            results: bt,
            trades: bt.all_trades || bt.trades || [],
            performance: bt.results || {},
            riskMetrics: bt.detailed_statistics || {},
            chartData: bt.chart_visualization || null,
            status: 'COMPLETED',
            startedAt: bt.started_at ? new Date(bt.started_at) : null,
            completedAt: bt.completed_at ? new Date(bt.completed_at) : new Date(),
          });

          await DatabaseService.recordUserEvent(user.id, 'BACKTEST_RUN', {
            backtestId: stored.id,
            market,
            strategyName: strategy,
            totalTrades: bt.total_trades || (bt.all_trades?.length ?? 0),
            winRate: bt.results?.win_rate ?? bt.win_rate ?? null,
            totalProfit: bt.results?.total_profit ?? bt.total_profit ?? null,
          });
        } catch (e) {
          console.warn('Failed to persist backtest results', e);
        }
      }

      // Update conversation activity
      await DatabaseService.updateConversationActivity(conversation.id);

      // Return response with conversation context
      const responseData: any = {
        ...aiResponse,
        conversationId: conversation.id,
        conversation: {
          id: conversation.id,
          title: conversation.title,
          messageCount: conversation.messageCount + 2, // +2 for the messages we just added
          lastActivityAt: new Date(),
        }
      };

      // Add visualization if generated
      if (visualization) {
        responseData.visualization = visualization;

        // Update the message to include visualization
        if (aiResponse.message) {
          aiResponse.message.visualization = visualization;
        }
      }

      return createSecureResponse(responseData, 200, user);

    } catch (error) {
      console.error('[api/chat] failed', error);

      if (error instanceof z.ZodError) {
        return createSecureResponse(
          { error: 'Validation failed', issues: error.issues },
          422
        );
      }

      // Record error analytics if user is available
      const user = (request as any).user;
      if (user) {
        await DatabaseService.recordUserEvent(user.id, 'CHAT_ERROR', {
          error: (error as Error).message,
          timestamp: new Date()
        });
      }

      return createSecureResponse(
        { error: 'Chat request failed' },
        500
      );
    }
  });
}

// Get conversation history
export async function GET(request: NextRequest) {
  try {
    // Require authentication
    const authResponse = await AuthMiddleware.requireAuth(request);
    if (authResponse) return authResponse;

    const user = (request as any).user;
    const { searchParams } = new URL(request.url);
    const conversationId = searchParams.get('conversationId');

    if (!conversationId) {
      // Return list of user's conversations
      const conversations = await DatabaseService.findConversationsByUser(user.id);
      return createSecureResponse({ conversations }, 200, user);
    }

    // Return specific conversation with messages
    const conversation = await DatabaseService.findConversationById(conversationId);
    if (!conversation || conversation.userId !== user.id) {
      return createSecureResponse(
        { error: 'Conversation not found or access denied' },
        404,
        user
      );
    }

    return createSecureResponse({ conversation }, 200, user);

  } catch (error) {
    console.error('[api/chat] GET failed', error);
    return createSecureResponse({ error: 'Failed to fetch conversations' }, 500);
  }
}
