/**
 * Subscription Management API Route
 * GET /api/billing/subscription - Get current subscription
 * PUT /api/billing/subscription - Update subscription (upgrade/downgrade)
 * DELETE /api/billing/subscription - Cancel subscription
 */

import { NextRequest } from 'next/server';
import { AuthMiddleware, createSecureResponse, DatabaseService } from '@/lib/auth';
import { StripeService, SUBSCRIPTION_PLANS, SubscriptionTier } from '@/lib/stripe';
import { ErrorHandler, logger, performanceMonitor } from '@/lib/error-handler';

export async function GET(request: NextRequest) {
  const endTimer = performanceMonitor.startTimer('get_subscription_api');

  try {
    // Require authentication
    const authResponse = await AuthMiddleware.requireAuth(request);
    if (authResponse) return authResponse;

    const user = (request as any).user;

    // Get user profile
    const userProfile = await DatabaseService.findUserById(user.id);
    if (!userProfile) {
      return createSecureResponse({ error: 'User not found' }, 404, user);
    }

    // Get subscription details from Stripe if customer exists
    let stripeSubscription = null;
    if (userProfile.stripeCustomerId) {
      try {
        const subscriptions = await StripeService.listCustomerSubscriptions(userProfile.stripeCustomerId);
        stripeSubscription = subscriptions.data[0] || null; // Get most recent
      } catch (error) {
        logger.warn('Failed to fetch Stripe subscription', { userId: user.id, error });
      }
    }

    // Return subscription info
    const sub = stripeSubscription as any;
    const response = createSecureResponse({
      subscription: {
        tier: userProfile.subscriptionTier,
        status: sub?.status || 'inactive',
        currentPeriodStart: sub?.current_period_start,
        currentPeriodEnd: sub?.current_period_end,
        cancelAtPeriodEnd: sub?.cancel_at_period_end,
        trialEnd: userProfile.trialEndsAt,
        stripeCustomerId: userProfile.stripeCustomerId,
        subscriptionId: sub?.id,
      },
      plan: SUBSCRIPTION_PLANS[userProfile.subscriptionTier as SubscriptionTier],
    }, 200, user);

    endTimer();
    return response;

  } catch (error) {
    endTimer();
    logger.error('Get subscription error', { error, userId: (request as any).user?.id });

    await ErrorHandler.handleError(error as Error, {
      url: request.url,
      method: request.method,
      userId: (request as any).user?.id,
    });

    return createSecureResponse(
      { error: 'Failed to fetch subscription' },
      500
    );
  }
}

export async function PUT(request: NextRequest) {
  const endTimer = performanceMonitor.startTimer('update_subscription_api');

  try {
    // Require authentication
    const authResponse = await AuthMiddleware.requireAuth(request);
    if (authResponse) return authResponse;

    const user = (request as any).user;

    // Parse request body
    const body = await request.json();
    const { newTier } = body;

    if (!newTier || !Object.keys(SUBSCRIPTION_PLANS).includes(newTier)) {
      return createSecureResponse(
        { error: 'Invalid subscription tier' },
        400,
        user
      );
    }

    // Get user profile
    const userProfile = await DatabaseService.findUserById(user.id);
    if (!userProfile) {
      return createSecureResponse({ error: 'User not found' }, 404, user);
    }

    // Check if user has an active subscription
    if (!userProfile.stripeCustomerId) {
      return createSecureResponse(
        { error: 'No active subscription found' },
        400,
        user
      );
    }

    // Get current subscription
    const subscriptions = await StripeService.listCustomerSubscriptions(userProfile.stripeCustomerId);
    const currentSubscription = subscriptions.data[0];

    if (!currentSubscription) {
      return createSecureResponse(
        { error: 'No active subscription found' },
        400,
        user
      );
    }

    const newPlanConfig = SUBSCRIPTION_PLANS[newTier as SubscriptionTier];
    if (!newPlanConfig.priceId) {
      return createSecureResponse(
        { error: 'New subscription tier not available' },
        400,
        user
      );
    }

    // Update subscription in Stripe
    await StripeService.updateSubscription(
      currentSubscription.id,
      newPlanConfig.priceId,
      user.id
    );

    // Record analytics
    await DatabaseService.recordUserEvent(user.id, 'SUBSCRIPTION_UPDATED', {
      oldTier: userProfile.subscriptionTier,
      newTier,
      subscriptionId: currentSubscription.id,
    });

    logger.info('Subscription updated', {
      userId: user.id,
      oldTier: userProfile.subscriptionTier,
      newTier,
    });

    const response = createSecureResponse({
      success: true,
      message: `Subscription updated to ${newTier}`,
      subscription: {
        tier: newTier,
        status: 'updating', // Will be updated via webhook
      },
    }, 200, user);

    endTimer();
    return response;

  } catch (error) {
    endTimer();
    logger.error('Update subscription error', { error, userId: (request as any).user?.id });

    await ErrorHandler.handleError(error as Error, {
      url: request.url,
      method: request.method,
      userId: (request as any).user?.id,
    });

    return createSecureResponse(
      { error: 'Failed to update subscription' },
      500
    );
  }
}

export async function DELETE(request: NextRequest) {
  const endTimer = performanceMonitor.startTimer('cancel_subscription_api');

  try {
    // Require authentication
    const authResponse = await AuthMiddleware.requireAuth(request);
    if (authResponse) return authResponse;

    const user = (request as any).user;

    // Get user profile
    const userProfile = await DatabaseService.findUserById(user.id);
    if (!userProfile || !userProfile.stripeCustomerId) {
      return createSecureResponse(
        { error: 'No active subscription found' },
        400,
        user
      );
    }

    // Get current subscription
    const subscriptions = await StripeService.listCustomerSubscriptions(userProfile.stripeCustomerId);
    const currentSubscription = subscriptions.data[0];

    if (!currentSubscription) {
      return createSecureResponse(
        { error: 'No active subscription found' },
        400,
        user
      );
    }

    // Cancel subscription (end of period)
    await StripeService.cancelSubscription(currentSubscription.id, user.id);

    // Record analytics
    const cs = currentSubscription as any;
    await DatabaseService.recordUserEvent(user.id, 'SUBSCRIPTION_CANCELLED', {
      subscriptionId: cs.id,
      tier: userProfile.subscriptionTier,
      cancelAt: cs.cancel_at,
    });

    logger.info('Subscription cancelled', {
      userId: user.id,
      subscriptionId: currentSubscription.id,
    });

    const response = createSecureResponse({
      success: true,
      message: 'Subscription will be cancelled at the end of the current billing period',
      cancelAt: cs.cancel_at,
    }, 200, user);

    endTimer();
    return response;

  } catch (error) {
    endTimer();
    logger.error('Cancel subscription error', { error, userId: (request as any).user?.id });

    await ErrorHandler.handleError(error as Error, {
      url: request.url,
      method: request.method,
      userId: (request as any).user?.id,
    });

    return createSecureResponse(
      { error: 'Failed to cancel subscription' },
      500
    );
  }
}
