/**
 * Create Subscription API Route
 * POST /api/billing/create-subscription
 * Creates a new Stripe subscription for a user
 */

import { NextRequest } from 'next/server';
import { z } from 'zod';
import { AuthMiddleware, createSecureResponse, DatabaseService } from '@/lib/auth';
import { StripeService, SUBSCRIPTION_PLANS, SubscriptionTier } from '@/lib/stripe';
import { ErrorHandler, logger, performanceMonitor } from '@/lib/error-handler';

const createSubscriptionSchema = z.object({
  tier: z.enum(['TRIAL', 'PRO', 'ELITE']),
  paymentMethodId: z.string().optional(), // For immediate payment
});

export async function POST(request: NextRequest) {
  const endTimer = performanceMonitor.startTimer('create_subscription_api');
  const requestId = `sub_${Date.now()}_${Math.random().toString(36).substring(2, 8)}`;

  try {
    // Require authentication
    const authResponse = await AuthMiddleware.requireAuth(request);
    if (authResponse) return authResponse;

    const user = (request as any).user;

    // Parse request body
    let body;
    try {
      body = await request.json();
    } catch (parseError) {
      throw new Error('Invalid JSON in request body');
    }

    // Validate with Zod
    const { tier, paymentMethodId } = createSubscriptionSchema.parse(body);

    // Check if user already has an active subscription
    const userProfile = await DatabaseService.findUserById(user.id);
    if (!userProfile) {
      throw new Error('User not found');
    }

    // Allow FREE tier users to upgrade to any plan
    // Allow other tiers to upgrade to ELITE only (for now)
    if (userProfile.subscriptionTier !== 'FREE' && tier !== 'ELITE') {
      return createSecureResponse(
        {
          error: 'User already has an active subscription',
          currentTier: userProfile.subscriptionTier,
          message: 'Contact support to change your subscription tier.'
        },
        400,
        user
      );
    }

    const planConfig = SUBSCRIPTION_PLANS[tier];
    if (!planConfig.priceId) {
      return createSecureResponse(
        { error: 'Subscription tier not available' },
        400,
        user
      );
    }

    // Create or get Stripe customer
    let customerId = userProfile.stripeCustomerId;
    if (!customerId) {
      customerId = await StripeService.createCustomer(
        user.id,
        user.email,
        `${user.firstName} ${user.lastName}`.trim()
      );

      // Update user with customer ID
      await DatabaseService.updateUser(user.id, {
        stripeCustomerId: customerId,
      });
    }

    // Attach payment method if provided
    if (paymentMethodId) {
      await StripeService.attachPaymentMethod(customerId, paymentMethodId);
    }

    // Create subscription
    const subscription = await StripeService.createSubscription(
      customerId,
      planConfig.priceId,
      user.id,
      tier === 'TRIAL' ? planConfig.trialDays : undefined
    );

    // Persist subscription linkage on the user record
    const sub = subscription as any;
    await DatabaseService.updateUser(user.id, {
      subscriptionId: sub.id,
      subscriptionTier: tier,
      trialEndsAt: sub.trial_end ? new Date(sub.trial_end * 1000) : undefined,
      subscriptionEndsAt: sub.current_period_end ? new Date(sub.current_period_end * 1000) : undefined,
      canceledAt: null,
    });

    // Record analytics
    await DatabaseService.recordUserEvent(user.id, 'SUBSCRIPTION_CREATED', {
      tier,
      subscriptionId: subscription.id,
      priceId: planConfig.priceId,
      requestId,
    });

    logger.info('Subscription created successfully', {
      userId: user.id,
      tier,
      subscriptionId: subscription.id,
      requestId,
    });

    // Return subscription details
    const response = createSecureResponse({
      success: true,
      subscription: {
        id: sub.id,
        status: sub.status,
        tier,
        currentPeriodStart: sub.current_period_start,
        currentPeriodEnd: sub.current_period_end,
        cancelAtPeriodEnd: sub.cancel_at_period_end,
        latestInvoice: sub.latest_invoice,
      },
      clientSecret: sub.latest_invoice?.payment_intent?.client_secret,
    }, 200, user);

    endTimer();
    return response;

  } catch (error) {
    endTimer();

    logger.error('Create subscription error', {
      error: (error as Error).message,
      userId: (request as any).user?.id,
      requestId,
    });

    // Handle different error types
    if (error instanceof z.ZodError) {
      return createSecureResponse(
        { error: 'Validation failed', details: error.issues },
        400
      );
    }

    await ErrorHandler.handleError(error as Error, {
      requestId,
      url: request.url,
      method: request.method,
      userId: (request as any).user?.id,
    });

    return createSecureResponse(
      { error: 'Failed to create subscription' },
      500
    );
  }
}
