/**
 * Create Payment Intent API Route
 * POST /api/billing/create-payment-intent
 * Creates a Stripe PaymentIntent for subscription signup
 * 
 * Security: Requires either valid auth token OR valid userId that matches email
 */

import { NextRequest, NextResponse } from 'next/server';
import Stripe from 'stripe';
import { SUBSCRIPTION_PLANS, SubscriptionTier } from '@/lib/stripe';
import { logger } from '@/lib/error-handler';
import { DatabaseService } from '@/lib/database';

// Lazy Stripe initialization to avoid build-time errors
let _stripe: Stripe | null = null;
function getStripe(): Stripe {
  if (!_stripe) {
    const secretKey = process.env.STRIPE_SECRET_KEY;
    if (!secretKey) {
      throw new Error('STRIPE_SECRET_KEY is not configured');
    }
    _stripe = new Stripe(secretKey, {
      apiVersion: '2024-12-18.acacia',
    });
  }
  return _stripe;
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { tier, email, userId, firstName, lastName } = body;

    if (!tier || !email) {
      return NextResponse.json(
        { error: 'Tier and email are required' },
        { status: 400 }
      );
    }

    // Security: Validate that userId exists and matches the email
    // This prevents unauthenticated creation of Stripe customers
    if (userId) {
      const user = await DatabaseService.findUserById(userId);
      if (!user || user.email.toLowerCase() !== email.toLowerCase()) {
        logger.warn('Payment intent creation blocked: userId/email mismatch', { userId, email });
        return NextResponse.json(
          { error: 'Invalid user credentials' },
          { status: 403 }
        );
      }
    } else {
      // If no userId, check for auth token
      const authHeader = request.headers.get('authorization');
      if (authHeader?.startsWith('Bearer ')) {
        const token = authHeader.substring(7);
        try {
          const decoded = await DatabaseService.verifyAuthToken(token);
          if (decoded.email?.toLowerCase() !== email.toLowerCase()) {
            return NextResponse.json(
              { error: 'Email mismatch with authenticated user' },
              { status: 403 }
            );
          }
        } catch {
          return NextResponse.json(
            { error: 'Invalid authentication token' },
            { status: 401 }
          );
        }
      } else {
        // No userId and no auth token - reject
        return NextResponse.json(
          { error: 'Authentication required' },
          { status: 401 }
        );
      }
    }

    // Validate tier
    const validTiers = ['FREE', 'TRIAL', 'PRO', 'ELITE'] as const;
    if (!validTiers.includes(tier.toUpperCase())) {
      return NextResponse.json(
        { error: 'Invalid subscription tier' },
        { status: 400 }
      );
    }

    const selectedTier = tier.toUpperCase() as SubscriptionTier;
    const plan = SUBSCRIPTION_PLANS[selectedTier];

    // For free tier, no payment required
    if (selectedTier === 'FREE' && plan.price === 0) {
      return NextResponse.json({
        requiresPayment: false,
        tier: selectedTier,
      });
    }

    // Create or retrieve customer
    const stripe = getStripe();
    let customer: Stripe.Customer;
    const existingCustomers = await stripe.customers.list({ email, limit: 1 });
    
    if (existingCustomers.data.length > 0) {
      customer = existingCustomers.data[0];
      // Update customer metadata if needed
      if (userId) {
        await stripe.customers.update(customer.id, {
          metadata: { userId },
          name: `${firstName || ''} ${lastName || ''}`.trim() || undefined,
        });
      }
    } else {
      customer = await stripe.customers.create({
        email,
        name: `${firstName || ''} ${lastName || ''}`.trim() || undefined,
        metadata: userId ? { userId } : {},
      });
    }

    // Determine amount based on tier and trial availability
    let amount: number;
    let description: string;
    const planConfig = plan as typeof plan & { trialPrice?: number; trialDays?: number };

    // Use trial price if available (for PRO and ELITE tiers)
    if (planConfig.trialPrice !== undefined && planConfig.trialDays && planConfig.trialDays > 0) {
      amount = planConfig.trialPrice * 100; // Convert to cents
      description = `${planConfig.trialDays}-Day Trial - Then $${plan.price}/month`;
    } else if (selectedTier === 'TRIAL') {
      amount = 100; // $1.00 for legacy trial tier
      description = '7-Day Trial - Then $20/month';
    } else {
      amount = plan.price * 100; // Convert to cents
      description = `${plan.name} Subscription - $${plan.price}/month`;
    }

    // Create PaymentIntent
    const paymentIntent = await stripe.paymentIntents.create({
      amount,
      currency: 'usd',
      customer: customer.id,
      description,
      metadata: {
        tier: selectedTier,
        userId: userId || '',
        email,
      },
      automatic_payment_methods: {
        enabled: true,
      },
    });

    logger.info('Created payment intent', {
      paymentIntentId: paymentIntent.id,
      tier: selectedTier,
      amount,
      customerId: customer.id,
    });

    return NextResponse.json({
      clientSecret: paymentIntent.client_secret,
      customerId: customer.id,
      amount,
      tier: selectedTier,
      requiresPayment: true,
    });

  } catch (error) {
    logger.error('Failed to create payment intent', { error });
    
    return NextResponse.json(
      { error: 'Failed to initialize payment' },
      { status: 500 }
    );
  }
}
