/**
 * Cancel Subscription API Route
 * POST /api/billing/cancel-subscription
 * Cancels user's active subscription at period end
 */

import { NextRequest, NextResponse } from 'next/server';
import { authenticateAndAuthorize, DatabaseService } from '@/lib/auth';
import { StripeService } from '@/lib/stripe';
import { SubscriptionTier } from '@/lib/types';
import { logger } from '@/lib/error-handler';

export async function POST(request: NextRequest) {
  try {
    // Require authentication
    const authResult = await authenticateAndAuthorize(request, [
      SubscriptionTier.FREE,
      SubscriptionTier.TRIAL,
      SubscriptionTier.PRO,
      SubscriptionTier.ELITE,
    ]);

    if (authResult.status !== 200) {
      return authResult.response;
    }

    const user = authResult.data?.user;
    if (!user) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }
    const body = await request.json().catch(() => ({}));
    const { reason, feedback } = body;

    // Get user's subscription info from database
    const dbUser = await DatabaseService.findUserById(user.id);
    
    if (!dbUser) {
      return NextResponse.json(
        { error: 'User not found' },
        { status: 404 }
      );
    }

    // Determine the Stripe subscription to cancel
    let subscriptionId = dbUser.subscriptionId;
    if (!subscriptionId && dbUser.stripeCustomerId) {
      const subs = await StripeService.listCustomerSubscriptions(dbUser.stripeCustomerId);
      subscriptionId = subs.data[0]?.id;
    }

    if (!subscriptionId) {
      return NextResponse.json({ error: 'No active subscription found' }, { status: 400 });
    }

    // Cancel the subscription in Stripe (at period end)
    const cancelledSubscription = await StripeService.cancelSubscription(subscriptionId, user.id);

    // Update user record
    await DatabaseService.updateUser(user.id, {
      canceledAt: new Date(),
      subscriptionId,
      subscriptionEndsAt: cancelledSubscription.current_period_end 
        ? new Date(cancelledSubscription.current_period_end * 1000)
        : undefined,
    });

    // Record the cancellation event with feedback
    await DatabaseService.recordUserEvent(user.id, 'SUBSCRIPTION_CANCELLED', {
      subscriptionId,
      reason: reason || 'not_provided',
      feedback: feedback || '',
      cancelAtPeriodEnd: true,
      currentPeriodEnd: cancelledSubscription.current_period_end,
    });

    logger.info('Subscription cancelled', {
      userId: user.id,
      subscriptionId,
      reason,
    });

    return NextResponse.json({
      success: true,
      message: 'Your subscription has been cancelled and will end at the current billing period.',
      endsAt: cancelledSubscription.current_period_end 
        ? new Date(cancelledSubscription.current_period_end * 1000).toISOString()
        : null,
    });

  } catch (error) {
    logger.error('Subscription cancellation error', { error });

    return NextResponse.json(
      { error: 'Failed to cancel subscription. Please try again or contact support.' },
      { status: 500 }
    );
  }
}
