/**
 * Betting Data API
 * 
 * Provides access to historical betting odds data for backtesting.
 * Used by the chat/RAG system to query sports betting data.
 * 
 * Endpoints:
 * GET /api/betting-data - Query betting data
 * GET /api/betting-data?action=catalog - Get data catalog for RAG
 * GET /api/betting-data?action=availability - Get data availability summary
 */

import { NextRequest, NextResponse } from 'next/server';
import { 
  queryBettingData, 
  getDataAvailability, 
  getDataCatalogForRAG,
  initializeBettingData,
  type QueryOptions 
} from '@/lib/sportsBettingData';

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const action = searchParams.get('action');

    // Initialize data on first request
    await initializeBettingData();

    // Handle different actions
    if (action === 'catalog') {
      // Return data catalog for RAG system
      const catalog = await getDataCatalogForRAG();
      return NextResponse.json({
        success: true,
        catalog,
        usage: 'Use this catalog to understand what betting data is available for backtesting.',
      });
    }

    if (action === 'availability') {
      // Return data availability summary
      const availability = await getDataAvailability();
      return NextResponse.json({
        success: true,
        availability,
        message: 'Data availability for all supported sports.',
      });
    }

    // Default: Query betting data
    const sport = searchParams.get('sport');
    
    if (!sport) {
      // Return general availability info
      const availability = await getDataAvailability();
      return NextResponse.json({
        success: true,
        message: 'Specify a sport to query data. Use ?sport=nba, ?sport=nfl, or ?sport=nhl',
        availableSports: Object.entries(availability).map(([key, info]) => ({
          sport: key,
          name: info.sport,
          games: info.totalGames,
          hasOdds: info.hasOdds,
          dateRange: info.dateRange,
        })),
        example: '/api/betting-data?sport=nba&season=2019&requireOdds=true',
      });
    }

    // Build query options
    const options: QueryOptions = {
      sport,
      startDate: searchParams.get('startDate') || undefined,
      endDate: searchParams.get('endDate') || undefined,
      season: searchParams.get('season') ? parseInt(searchParams.get('season')!) : undefined,
      team: searchParams.get('team') || undefined,
      requireOdds: searchParams.get('requireOdds') === 'true',
      limit: searchParams.get('limit') ? parseInt(searchParams.get('limit')!) : 100,
    };

    // Query the data
    const result = await queryBettingData(options);

    return NextResponse.json({
      success: result.success,
      message: result.message,
      query: options,
      availability: result.availability,
      warnings: result.warnings,
      totalGames: result.games.length,
      games: result.games,
    });

  } catch (error) {
    console.error('[api/betting-data] Error:', error);
    return NextResponse.json(
      { 
        success: false,
        error: 'Failed to query betting data',
        message: error instanceof Error ? error.message : 'Unknown error',
      },
      { status: 500 }
    );
  }
}

/**
 * POST endpoint for complex queries or backtest requests
 */
export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    
    // Initialize data
    await initializeBettingData();

    const { action, query } = body;

    if (action === 'backtest') {
      // Handle backtest request
      const { sport, strategy, options } = query;
      
      if (!sport || !strategy) {
        return NextResponse.json({
          success: false,
          error: 'Backtest requires sport and strategy parameters',
        }, { status: 400 });
      }

      // For now, return the data that would be used for backtesting
      // The actual strategy execution happens in the chat/RAG system
      const result = await queryBettingData({
        sport,
        startDate: options?.startDate,
        endDate: options?.endDate,
        season: options?.season,
        requireOdds: true,
        limit: options?.limit || 1000,
      });

      return NextResponse.json({
        success: result.success,
        message: result.message,
        backtestData: {
          sport,
          totalGames: result.games.length,
          gamesWithOdds: result.availability.gamesWithOdds,
          dateRange: result.availability.dateRange,
          warnings: result.warnings,
        },
        games: result.games,
      });
    }

    if (action === 'query') {
      // Handle data query
      const result = await queryBettingData(query);
      return NextResponse.json({
        success: result.success,
        message: result.message,
        availability: result.availability,
        warnings: result.warnings,
        totalGames: result.games.length,
        games: result.games,
      });
    }

    return NextResponse.json({
      success: false,
      error: 'Unknown action. Use "query" or "backtest"',
    }, { status: 400 });

  } catch (error) {
    console.error('[api/betting-data] POST Error:', error);
    return NextResponse.json(
      { 
        success: false,
        error: 'Failed to process request',
        message: error instanceof Error ? error.message : 'Unknown error',
      },
      { status: 500 }
    );
  }
}
