import { NextRequest, NextResponse } from 'next/server';
import { AuthMiddleware, DatabaseService } from '@/lib/auth';

function safeJsonParse<T>(value: string, fallback: T): T {
  try {
    return JSON.parse(value) as T;
  } catch {
    return fallback;
  }
}

function extractSummary(raw: any) {
  const resultsObj = raw?.results || raw || {};
  const totalTrades = raw?.total_trades ?? resultsObj?.total_trades ?? 0;
  const winRate = resultsObj?.win_rate ?? raw?.win_rate ?? null;
  const totalProfit = resultsObj?.total_profit ?? raw?.total_profit ?? null;

  return {
    totalTrades,
    winRate,
    totalProfit,
  };
}

export async function GET(request: NextRequest) {
  try {
    const authResponse = await AuthMiddleware.requireAuth(request);
    if (authResponse) return authResponse;

    const user = (request as any).user;
    const url = new URL(request.url);
    const limit = Math.min(100, Math.max(1, Number(url.searchParams.get('limit') || 50)));

    const backtests = await DatabaseService.findBacktestsByUser(user.id, limit);

    const items = backtests.map((bt) => {
      const rawResults = safeJsonParse<any>(bt.results, {});
      const summary = extractSummary(rawResults);

      return {
        id: bt.id,
        jobId: bt.jobId,
        market: bt.market,
        strategyName: bt.strategyName,
        status: bt.status,
        createdAt: bt.createdAt,
        executionTime: bt.executionTime,
        summary,
      };
    });

    return NextResponse.json({ backtests: items, total: items.length });
  } catch (error) {
    console.error('[api/backtests] GET failed', error);
    return NextResponse.json({ error: 'Failed to load backtests' }, { status: 500 });
  }
}


