/**
 * Backtest Job Status API
 * GET /api/backtest/job?jobId=xxx - Get job status
 * POST /api/backtest/job - Submit new backtest job
 */

import { NextRequest, NextResponse } from 'next/server';
import {
    addBacktestJob,
    getJobStatus,
    getUserJobs,
    getQueueStats
} from '@/lib/queue/BacktestQueue';
import { AuthMiddleware } from '@/lib/auth';

// Extract user ID from request
function getUserIdFromRequest(request: NextRequest): string {
    const authHeader = request.headers.get('authorization');
    if (authHeader) {
        return `user-${authHeader.substring(0, 16)}`;
    }
    // Try from cookies
    const sessionCookie = request.cookies.get('next-auth.session-token')?.value;
    if (sessionCookie) {
        return `session-${sessionCookie.substring(0, 16)}`;
    }
    return 'anonymous';
}

/**
 * GET - Get job status or user's recent jobs
 */
export async function GET(request: NextRequest) {
    try {
        const authResponse = await AuthMiddleware.requireAuth(request);
        if (authResponse) return authResponse;

        const userId = getUserIdFromRequest(request);

        const { searchParams } = new URL(request.url);
        const jobId = searchParams.get('jobId');

        // If jobId provided, get specific job status
        if (jobId) {
            const status = await getJobStatus(jobId);
            return NextResponse.json({
                success: true,
                job: {
                    id: jobId,
                    ...status,
                },
            });
        }

        // Otherwise, get user's recent jobs and queue stats
        const [userJobs, stats] = await Promise.all([
            getUserJobs(userId, 20),
            getQueueStats(),
        ]);

        return NextResponse.json({
            success: true,
            jobs: userJobs.map(job => ({
                id: job.id,
                data: job.data,
                status: job.returnvalue ? 'completed' : (job.failedReason ? 'failed' : 'pending'),
                progress: job.progress,
                createdAt: job.data.createdAt,
                result: job.returnvalue,
                error: job.failedReason,
            })),
            queueStats: stats,
        });

    } catch (error) {
        console.error('Job status API error:', error);
        return NextResponse.json(
            { error: 'Failed to get job status' },
            { status: 500 }
        );
    }
}

/**
 * POST - Submit a new backtest job
 */
export async function POST(request: NextRequest) {
    try {
        const authResponse = await AuthMiddleware.requireAuth(request);
        if (authResponse) return authResponse;

        const userId = getUserIdFromRequest(request);

        const body = await request.json();
        const { market, strategyName, strategyCode, parameters, season } = body;

        // Validate required fields
        if (!market) {
            return NextResponse.json(
                { error: 'Market is required' },
                { status: 400 }
            );
        }

        // Create job
        const job = await addBacktestJob({
            userId,
            market,
            strategyName: strategyName || 'dynamic_custom',
            strategyCode,
            parameters: parameters || {},
            season,
        });

        return NextResponse.json({
            success: true,
            message: 'Backtest job queued successfully',
            job: {
                id: job.id,
                queuePosition: await job.getState() === 'waiting' ?
                    (await job.queue.getWaitingCount()) : 0,
            },
        });

    } catch (error) {
        console.error('Job submit API error:', error);
        return NextResponse.json(
            { error: 'Failed to submit backtest job' },
            { status: 500 }
        );
    }
}
