/**
 * Email Verification API Route
 * POST /api/auth/verify-email
 * Verifies user email with token
 */

import { NextRequest } from 'next/server';
import { DatabaseService, createSecureResponse } from '@/lib/auth';
import { ErrorHandler, logger } from '@/lib/error-handler';

export async function POST(request: NextRequest) {
  try {
    const { token } = await request.json();

    if (!token) {
      return createSecureResponse(
        { error: 'Verification token is required' },
        400
      );
    }

    // Find user with matching verification token
    const user = await DatabaseService.findUserByVerificationToken(token);

    if (!user) {
      return createSecureResponse(
        { error: 'Invalid or expired verification token' },
        400
      );
    }

    // Check if token has expired
    if (user.emailVerificationExpires && user.emailVerificationExpires < new Date()) {
      return createSecureResponse(
        { error: 'Verification token has expired' },
        400
      );
    }

    // Update user as verified and clear verification fields
    await DatabaseService.updateUser(user.id, {
      emailVerified: true,
      emailVerifiedAt: new Date(),
      emailVerificationToken: null,
      emailVerificationExpires: null,
    });

    // Record verification event
    await DatabaseService.recordUserEvent(user.id, 'EMAIL_VERIFIED', {
      method: 'token_verification',
    });

    logger.info('Email verified successfully', { userId: user.id });

    return createSecureResponse({
      success: true,
      message: 'Email verified successfully',
      user: {
        id: user.id,
        email: user.email,
        emailVerified: true,
      },
    });

  } catch (error) {
    logger.error('Email verification error', { error });

    await ErrorHandler.handleError(error as Error, {
      url: request.url,
      method: request.method,
    });

    return createSecureResponse(
      { error: 'Failed to verify email' },
      500
    );
  }
}

// Resend verification email
export async function PUT(request: NextRequest) {
  try {
    const { email } = await request.json();

    if (!email) {
      return createSecureResponse(
        { error: 'Email is required' },
        400
      );
    }

    // Find user by email
    const user = await DatabaseService.findUserByEmail(email);

    if (!user) {
      // Don't reveal if email exists or not for security
      return createSecureResponse(
        { success: true, message: 'If an account exists, verification email has been sent' }
      );
    }

    if (user.emailVerified) {
      return createSecureResponse(
        { error: 'Email is already verified' },
        400
      );
    }

    // Generate new verification token
    const verificationToken = require('crypto').randomBytes(32).toString('hex');
    const expiresAt = new Date(Date.now() + 24 * 60 * 60 * 1000); // 24 hours

    // Update user with new token
    await DatabaseService.updateUser(user.id, {
      emailVerificationToken: verificationToken,
      emailVerificationExpires: expiresAt,
    });

    // Send verification email
    const { EmailUtils } = await import('@/lib/email');
    const emailSent = await EmailUtils.sendEmailVerification(
      { email: user.email, firstName: user.firstName || 'User' },
      verificationToken
    );

    if (!emailSent) {
      logger.warn('Failed to send verification email', { userId: user.id });
      return createSecureResponse(
        { error: 'Failed to send verification email' },
        500
      );
    }

    // Record resend event
    await DatabaseService.recordUserEvent(user.id, 'VERIFICATION_EMAIL_RESENT', {
      method: 'resend_request',
    });

    return createSecureResponse({
      success: true,
      message: 'Verification email sent successfully',
    });

  } catch (error) {
    logger.error('Resend verification email error', { error });

    await ErrorHandler.handleError(error as Error, {
      url: request.url,
      method: request.method,
    });

    return createSecureResponse(
      { error: 'Failed to resend verification email' },
      500
    );
  }
}
