/**
 * User Signup API Route
 * POST /api/auth/signup
 */

import { NextRequest, NextResponse } from 'next/server';
import { DatabaseService } from '@/lib/database';
import { EmailUtils } from '@/lib/email';
import bcrypt from 'bcryptjs';
import jwt from 'jsonwebtoken';

export async function POST(request: NextRequest) {
  try {
    // Parse request body
    const body = await request.json();
    const { email, password, firstName, lastName, subscriptionTier = 'FREE' } = body;

    // Basic validation
    if (!email || !password || !firstName || !lastName) {
      return NextResponse.json(
        { error: 'Missing required fields', details: ['email', 'password', 'firstName', 'lastName are required'] },
        { status: 400 }
      );
    }

    // Email validation
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(email)) {
      return NextResponse.json(
        { error: 'Invalid email format' },
        { status: 400 }
      );
    }

    // Password validation
    if (password.length < 8) {
      return NextResponse.json(
        { error: 'Password must be at least 8 characters' },
        { status: 400 }
      );
    }

    // Check if user already exists
    const existingUser = await DatabaseService.findUserByEmail(email);
    if (existingUser) {
      return NextResponse.json(
        { error: 'User already exists with this email address' },
        { status: 409 }
      );
    }

    // Generate email verification token
    const verificationToken = DatabaseService.generateVerificationToken();
    const verificationExpires = new Date(Date.now() + 24 * 60 * 60 * 1000); // 24 hours

    // Create new user
    const newUser = await DatabaseService.createUser({
      email,
      password,
      firstName,
      lastName,
      subscriptionTier: 'FREE', // Force FREE tier to prevent mass assignment exploit
      emailVerificationToken: verificationToken,
      emailVerificationExpires: verificationExpires,
      emailVerified: true, // Auto-verify for critical UX fix (allow immediate access)
    });

    // Send verification email (non-blocking)
    try {
      await EmailUtils.sendEmailVerification(
        { email: newUser.email, firstName: newUser.firstName },
        verificationToken
      );
      console.log(`✅ Verification email sent to ${newUser.email}`);
    } catch (emailError) {
      // Log but don't block signup if email fails
      console.error('Failed to send verification email:', emailError);
    }

    // Generate JWT token
    const jwtSecret = process.env.JWT_SECRET;
    if (!jwtSecret) {
      console.error('JWT_SECRET not configured');
      return NextResponse.json({ error: 'Server configuration error' }, { status: 500 });
    }

    const token = jwt.sign(
      {
        userId: newUser.id,
        email: newUser.email,
        subscriptionTier: newUser.subscriptionTier,
      },
      jwtSecret,
      { expiresIn: '7d' }
    );

    // Create response
    const response = NextResponse.json({
      success: true,
      user: {
        id: newUser.id,
        email: newUser.email,
        firstName: newUser.firstName,
        lastName: newUser.lastName,
        subscriptionTier: newUser.subscriptionTier,
        emailVerified: newUser.emailVerified,
        createdAt: newUser.createdAt,
      },
      token,
      message: 'Account created successfully. Please check your email for verification instructions.',
    }, { status: 201 });

    // Set HTTP-only cookie
    response.cookies.set('auth-token', token, {
      httpOnly: true,
      secure: process.env.NODE_ENV === 'production',
      sameSite: 'strict',
      maxAge: 7 * 24 * 60 * 60, // 7 days
    });

    return response;

  } catch (error) {
    console.error('Signup error:', error);
    const errorMessage = error instanceof Error ? error.message : 'Unknown error';
    return NextResponse.json(
      { error: 'Registration failed', details: errorMessage },
      { status: 500 }
    );
  }
}
