/**
 * Password Reset API Route
 * POST /api/auth/reset-password - Request password reset
 * PUT /api/auth/reset-password - Reset password with token
 */

import { NextRequest } from 'next/server';
import { z } from 'zod';
import { DatabaseService, createSecureResponse } from '@/lib/auth';
import { ErrorHandler, logger } from '@/lib/error-handler';

const requestResetSchema = z.object({
  email: z.string().email('Invalid email format'),
});

const resetPasswordSchema = z.object({
  token: z.string().min(1, 'Reset token is required'),
  password: z.string().min(8, 'Password must be at least 8 characters'),
  confirmPassword: z.string().min(8, 'Confirm password is required'),
});

// Request password reset
export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { email } = requestResetSchema.parse(body);

    // Find user by email
    const user = await DatabaseService.findUserByEmail(email);

    if (!user) {
      // Don't reveal if email exists or not for security
      return createSecureResponse({
        success: true,
        message: 'If an account exists, password reset instructions have been sent'
      });
    }

    // Generate reset token
    const resetToken = require('crypto').randomBytes(32).toString('hex');
    const expiresAt = new Date(Date.now() + 60 * 60 * 1000); // 1 hour

    // Update user with reset token
    await DatabaseService.updateUser(user.id, {
      passwordResetToken: resetToken,
      passwordResetExpires: expiresAt,
    });

    // Send password reset email
    const { EmailUtils } = await import('@/lib/email');
    const emailSent = await EmailUtils.sendPasswordReset(
      { email: user.email, firstName: user.firstName || 'User' },
      resetToken
    );

    if (!emailSent) {
      logger.warn('Failed to send password reset email', { userId: user.id });
      return createSecureResponse(
        { error: 'Failed to send password reset email' },
        500
      );
    }

    // Record password reset request
    await DatabaseService.recordUserEvent(user.id, 'PASSWORD_RESET_REQUESTED', {
      method: 'email_reset',
    });

    logger.info('Password reset email sent', { userId: user.id });

    return createSecureResponse({
      success: true,
      message: 'Password reset instructions sent to your email',
    });

  } catch (error) {
    logger.error('Password reset request error', { error });

    if (error instanceof z.ZodError) {
      return createSecureResponse(
        { error: 'Validation failed', details: error.issues },
        400
      );
    }

    await ErrorHandler.handleError(error as Error, {
      url: request.url,
      method: request.method,
    });

    return createSecureResponse(
      { error: 'Failed to process password reset request' },
      500
    );
  }
}

// Reset password with token
export async function PUT(request: NextRequest) {
  try {
    const body = await request.json();
    const { token, password, confirmPassword } = resetPasswordSchema.parse(body);

    // Validate passwords match
    if (password !== confirmPassword) {
      return createSecureResponse(
        { error: 'Passwords do not match' },
        400
      );
    }

    // Find user with reset token
    const user = await DatabaseService.findUserByPasswordResetToken(token);

    if (!user) {
      return createSecureResponse(
        { error: 'Invalid or expired reset token' },
        400
      );
    }

    // Check if token has expired
    if (user.passwordResetExpires && user.passwordResetExpires < new Date()) {
      return createSecureResponse(
        { error: 'Reset token has expired' },
        400
      );
    }

    // Hash new password and update user
    const hashedPassword = await DatabaseService.hashPassword(password);
    await DatabaseService.updateUser(user.id, {
      passwordHash: hashedPassword,
      passwordResetToken: null,
      passwordResetExpires: null,
    });

    // Record password reset
    await DatabaseService.recordUserEvent(user.id, 'PASSWORD_RESET_COMPLETED', {
      method: 'token_reset',
    });

    logger.info('Password reset completed', { userId: user.id });

    return createSecureResponse({
      success: true,
      message: 'Password reset successfully',
    });

  } catch (error) {
    logger.error('Password reset error', { error });

    if (error instanceof z.ZodError) {
      return createSecureResponse(
        { error: 'Validation failed', details: error.issues },
        400
      );
    }

    await ErrorHandler.handleError(error as Error, {
      url: request.url,
      method: request.method,
    });

    return createSecureResponse(
      { error: 'Failed to reset password' },
      500
    );
  }
}
