/**
 * User Profile API Route
 * GET /api/auth/profile - Get current user profile
 * PUT /api/auth/profile - Update user profile
 */

import { NextRequest } from 'next/server';
import { AuthMiddleware, SecurityMiddleware, createSecureResponse, DatabaseService } from '@/lib/auth';
import { z } from 'zod';

function parseJsonArray(value: any): string[] | undefined {
  if (!value) return undefined;
  if (Array.isArray(value)) return value.map(String);
  if (typeof value !== 'string') return undefined;
  try {
    const parsed = JSON.parse(value);
    if (Array.isArray(parsed)) return parsed.map(String);
    return undefined;
  } catch {
    return undefined;
  }
}

function getSubscriptionStatus(profile: any): string {
  const now = new Date();
  const tier = (profile?.subscriptionTier || 'FREE').toString().toUpperCase();

  // Treat FREE as trial if trialEndsAt is set and not expired
  if (tier === 'FREE') {
    if (profile?.trialEndsAt && new Date(profile.trialEndsAt) > now) return 'TRIAL';
    return 'FREE';
  }

  if (profile?.canceledAt) return 'CANCELLED';
  if (profile?.subscriptionEndsAt && new Date(profile.subscriptionEndsAt) < now) return 'EXPIRED';
  return 'ACTIVE';
}

const updateProfileSchema = z.object({
  firstName: z.string().min(1).max(50).optional(),
  lastName: z.string().min(1).max(50).optional(),
  riskTolerance: z.enum(['LOW', 'MEDIUM', 'HIGH']).optional(),
  experienceLevel: z.enum(['BEGINNER', 'INTERMEDIATE', 'ADVANCED', 'EXPERT']).optional(),
  preferredMarkets: z.array(z.enum(['sports', 'crypto', 'stocks', 'forex'])).optional(),
  favoriteStrategies: z.array(z.string()).optional(),
});

export async function GET(request: NextRequest) {
  try {
    // Require authentication
    const authResponse = await AuthMiddleware.requireAuth(request);
    if (authResponse) return authResponse;

    const user = (request as any).user;

    // Get full user profile with stats
    const profile = await DatabaseService.findUserById(user.id);
    if (!profile) {
      return createSecureResponse({ error: 'User not found' }, 404);
    }

    // Get user statistics
    const stats = await DatabaseService.getUserStats(user.id);

    return createSecureResponse({
      user: {
        id: profile.id,
        email: profile.email,
        firstName: profile.firstName,
        lastName: profile.lastName,
        subscriptionTier: profile.subscriptionTier,
        subscriptionStatus: getSubscriptionStatus(profile),
        subscriptionId: profile.subscriptionId,
        stripeCustomerId: profile.stripeCustomerId,
        trialEndsAt: profile.trialEndsAt,
        subscriptionEndsAt: profile.subscriptionEndsAt,
        canceledAt: profile.canceledAt,
        emailVerified: profile.emailVerified,
        riskTolerance: profile.riskTolerance,
        experienceLevel: profile.experienceLevel,
        preferredMarkets: parseJsonArray(profile.preferredMarkets),
        favoriteStrategies: parseJsonArray(profile.favoriteStrategies),
        lastLoginAt: profile.lastLoginAt,
        createdAt: profile.createdAt,
        updatedAt: profile.updatedAt,
      },
      stats,
    });

  } catch (error) {
    console.error('Profile fetch error:', error);
    return createSecureResponse({ error: 'Internal server error' }, 500);
  }
}

export async function PUT(request: NextRequest) {
  try {
    // Require authentication
    const authResponse = await AuthMiddleware.requireAuth(request);
    if (authResponse) return authResponse;

    const user = (request as any).user;

    // Parse request body
    const body = await request.json();
    const validation = SecurityMiddleware.validateInput(body, {
      stringFields: ['firstName', 'lastName', 'riskTolerance', 'experienceLevel'],
      maxLength: {
        firstName: 50,
        lastName: 50
      }
    });

    if (!validation.valid) {
      return createSecureResponse(
        { error: 'Validation failed', details: validation.errors },
        400
      );
    }

    // Validate with Zod
    const updateData = updateProfileSchema.parse(body);

    // Update user profile
    const updatedUser = await DatabaseService.updateUser(user.id, updateData);

    // Record profile update event
    await DatabaseService.recordUserEvent(user.id, 'PROFILE_UPDATED', {
      fields: Object.keys(updateData)
    });

    return createSecureResponse({
      success: true,
      user: {
        id: updatedUser.id,
        email: updatedUser.email,
        firstName: updatedUser.firstName,
        lastName: updatedUser.lastName,
        subscriptionTier: updatedUser.subscriptionTier,
        subscriptionStatus: getSubscriptionStatus(updatedUser),
        subscriptionId: updatedUser.subscriptionId,
        stripeCustomerId: updatedUser.stripeCustomerId,
        trialEndsAt: updatedUser.trialEndsAt,
        subscriptionEndsAt: updatedUser.subscriptionEndsAt,
        canceledAt: updatedUser.canceledAt,
        emailVerified: updatedUser.emailVerified,
        riskTolerance: updatedUser.riskTolerance,
        experienceLevel: updatedUser.experienceLevel,
        preferredMarkets: parseJsonArray(updatedUser.preferredMarkets),
        favoriteStrategies: parseJsonArray(updatedUser.favoriteStrategies),
        updatedAt: updatedUser.updatedAt,
      },
    });

  } catch (error) {
    console.error('Profile update error:', error);

    if (error instanceof z.ZodError) {
      return createSecureResponse(
        { error: 'Validation failed', details: error.issues },
        400
      );
    }

    return createSecureResponse({ error: 'Internal server error' }, 500);
  }
}
