/**
 * Logout API Route
 * POST /api/auth/logout
 * Clears auth cookies and invalidates session
 */

import { NextRequest, NextResponse } from 'next/server';
import { DatabaseService, createSecureResponse, AuthService } from '@/lib/auth';
import { logger } from '@/lib/error-handler';

export async function POST(request: NextRequest) {
  try {
    // Get current user if authenticated
    const authResult = await AuthService.authenticateRequest(request);
    
    // Record logout event if user was authenticated
    if (authResult.user) {
      await DatabaseService.recordUserEvent(authResult.user.id, 'LOGOUT', {
        method: 'manual_logout',
      });
      logger.info('User logged out', { userId: authResult.user.id });
    }

    // Create response that clears the auth cookie
    const response = NextResponse.json({
      success: true,
      message: 'Logged out successfully',
    });

    // Clear the auth-token cookie
    response.cookies.set('auth-token', '', {
      httpOnly: true,
      secure: process.env.NODE_ENV === 'production',
      sameSite: 'lax',
      maxAge: 0, // Immediately expire
      path: '/',
    });

    return response;

  } catch (error) {
    logger.error('Logout error', { error });
    
    // Still clear cookies even on error
    const response = NextResponse.json({
      success: true,
      message: 'Logged out',
    });

    response.cookies.set('auth-token', '', {
      httpOnly: true,
      secure: process.env.NODE_ENV === 'production',
      sameSite: 'lax',
      maxAge: 0,
      path: '/',
    });

    return response;
  }
}
