/**
 * User Login API Route
 * POST /api/auth/login
 */

import { NextRequest } from 'next/server';
import { DatabaseService, AuthService, SecurityMiddleware, createSecureResponse } from '@/lib/auth';
import { ErrorHandler, ErrorFactory, logger, performanceMonitor } from '@/lib/error-handler';
import { z } from 'zod';

const loginSchema = z.object({
  email: z.string().email('Invalid email format'),
  password: z.string().min(6, 'Password must be at least 6 characters'),
});

export async function POST(request: NextRequest) {
  const endTimer = performanceMonitor.startTimer('login_api');
  const requestId = `login_${Date.now()}_${Math.random().toString(36).substring(2, 8)}`;

  try {
    // Rate limiting
    const rateLimitResponse = SecurityMiddleware.rateLimit(request, 5, 15 * 60 * 1000); // 5 attempts per 15 minutes
    if (rateLimitResponse) {
      endTimer();
      return rateLimitResponse;
    }

    // Parse request body
    let body;
    try {
      body = await request.json();
    } catch (parseError) {
      throw ErrorFactory.validation('Invalid JSON in request body');
    }

    const validation = SecurityMiddleware.validateInput(body, {
      required: ['email', 'password'],
      stringFields: ['email', 'password'],
      emailFields: ['email'],
      maxLength: { password: 128 }
    });

    if (!validation.valid) {
      throw ErrorFactory.validation('Input validation failed', {
        requestId,
        details: validation.errors
      });
    }

    // Validate with Zod
    const { email, password } = loginSchema.parse(body);

    // Find user by email
    const user = await DatabaseService.findUserByEmail(email);
    if (!user) {
      logger.warn('Login attempt with non-existent email', {
        email: email.substring(0, 3) + '***', // Partial email for logging
        requestId,
        ip: (request as any).ip
      });
      throw ErrorFactory.authentication('Invalid email or password');
    }

    // Verify password
    const isValidPassword = await DatabaseService.verifyPassword(password, user.passwordHash);
    if (!isValidPassword) {
      // Record failed login attempt
      await DatabaseService.recordUserEvent(user.id, 'LOGIN_FAILED', {
        reason: 'invalid_password',
        requestId,
        ip: (request as any).ip
      });

      logger.warn('Failed login attempt', {
        userId: user.id,
        requestId,
        ip: (request as any).ip
      });

      throw ErrorFactory.authentication('Invalid email or password');
    }

    // Check if email is verified (skip for now, implement later)
    if (!user.emailVerified) {
      throw ErrorFactory.authorization('Email not verified', {
        userId: user.id,
        requestId
      });
    }

    // Update last login
    await DatabaseService.updateUserLastLogin(user.id);

    // Record successful login
    await DatabaseService.recordUserEvent(user.id, 'LOGIN_SUCCESS', {
      userAgent: request.headers.get('user-agent') || undefined,
      ip: (request as any).ip,
      requestId
    });

    // Generate JWT token
    const authUser = {
      id: user.id,
      email: user.email,
      subscriptionTier: user.subscriptionTier,
      firstName: user.firstName || undefined,
      lastName: user.lastName || undefined,
    };

    const token = AuthService.generateToken(authUser);

    logger.info('Successful login', {
      userId: user.id,
      subscriptionTier: user.subscriptionTier,
      requestId
    });

    // Return success response
    const response = createSecureResponse({
      success: true,
      user: {
        id: user.id,
        email: user.email,
        firstName: user.firstName,
        lastName: user.lastName,
        subscriptionTier: user.subscriptionTier,
        emailVerified: user.emailVerified,
        lastLoginAt: user.lastLoginAt,
      },
      token,
    }, 200, authUser);

    // Set HTTP-only cookie for additional security
    response.cookies.set('auth-token', token, {
      httpOnly: true,
      secure: process.env.NODE_ENV === 'production',
      sameSite: 'strict',
      maxAge: 7 * 24 * 60 * 60, // 7 days
    });

    endTimer();
    return response;

  } catch (error) {
    endTimer();

    // Handle different error types
    if (error instanceof z.ZodError) {
      return createSecureResponse(
        { error: 'Validation failed', details: error.issues },
        400
      );
    }

    // Use our error handler for app errors
    if (error instanceof Error && 'statusCode' in error) {
      const appError = error as any;
      return createSecureResponse(
        { error: appError.message },
        appError.statusCode
      );
    }

    // Handle unexpected errors
    await ErrorHandler.handleError(error as Error, {
      requestId,
      url: request.url,
      method: request.method,
      userAgent: request.headers.get('user-agent') || undefined,
      ip: (request as any).ip,
    });

    return createSecureResponse(
      { error: 'Internal server error' },
      500
    );
  }
}
