/**
 * Announcements API Route
 * GET /api/announcements
 * Returns platform announcements and updates
 */

import { NextRequest } from 'next/server';
import { AuthMiddleware, createSecureResponse } from '@/lib/auth';
import { logger } from '@/lib/error-handler';
import { prisma } from '@/lib/database';

interface Announcement {
  id: string;
  title: string;
  date: string;
  content: string;
  type: 'feature' | 'update' | 'security' | 'performance' | 'maintenance';
  important: boolean;
  published: boolean;
}

function safeParseJson<T>(value: string | null | undefined, fallback: T): T {
  if (!value) return fallback;
  try {
    return JSON.parse(value) as T;
  } catch {
    return fallback;
  }
}

export async function GET(request: NextRequest) {
  try {
    // Optional authentication - allow anonymous access to announcements
    await AuthMiddleware.optionalAuth(request);
    const user = (request as any).user;

    // Announcements are stored as SystemAnalytics events with eventType = 'ANNOUNCEMENT'
    // eventData should be JSON: { title, date?, content, type, important, published }
    const rows = await prisma.systemAnalytics.findMany({
      where: { eventType: 'ANNOUNCEMENT' },
      orderBy: { createdAt: 'desc' },
      take: 50,
    });

    const announcements: Announcement[] = rows.map((row) => {
      const event = safeParseJson<any>(row.eventData, {});
      const published = event.published !== false; // default true
      const date = event.date || row.createdAt.toISOString().split('T')[0];

      return {
        id: row.id,
        title: event.title || 'Platform update',
        date,
        content: event.content || '',
        type: (event.type || 'update') as Announcement['type'],
        important: !!event.important,
        published,
      };
    });

    // Filter only published announcements and sort by date (newest first)
    const publishedAnnouncements = announcements
      .filter((announcement) => announcement.published)
      .sort((a, b) => new Date(b.date).getTime() - new Date(a.date).getTime());

    // Compute unread important count for signed-in users (using user_analytics reads)
    let unreadCount = publishedAnnouncements.filter(a => a.important).length;
    if (user) {
      const readEvents = await prisma.userAnalytics.findMany({
        where: {
          userId: user.id,
          eventType: 'ANNOUNCEMENT_READ',
        },
        select: { eventData: true },
        orderBy: { createdAt: 'desc' },
        take: 500,
      });

      const readIds = new Set<string>();
      for (const e of readEvents) {
        const payload = safeParseJson<any>(e.eventData, {});
        if (payload?.announcementId) readIds.add(String(payload.announcementId));
      }

      unreadCount = publishedAnnouncements.filter(a => a.important && !readIds.has(String(a.id))).length;
    }

    // Add metadata
    const response = {
      announcements: publishedAnnouncements,
      total: publishedAnnouncements.length,
      lastUpdated: new Date().toISOString(),
      unreadCount,
    };

    logger.debug('Announcements fetched', {
      count: publishedAnnouncements.length,
      important: response.unreadCount
    });

    return createSecureResponse(response, 200);

  } catch (error) {
    console.error('[api/announcements] failed', error);
    return createSecureResponse(
      { error: 'Failed to fetch announcements' },
      500
    );
  }
}
