import { NextRequest, NextResponse } from 'next/server';
import { AuthMiddleware, APIKeyValidator } from '@/lib/auth';

async function checkUrlHealthy(url: string, timeoutMs: number = 3000): Promise<boolean> {
  try {
    const controller = new AbortController();
    const id = setTimeout(() => controller.abort(), timeoutMs);
    const res = await fetch(url, { signal: controller.signal });
    clearTimeout(id);
    return res.ok;
  } catch {
    return false;
  }
}

export async function GET(request: NextRequest) {
  try {
    const authResponse = await AuthMiddleware.requireRole(request, 'ADMIN');
    if (authResponse) return authResponse;

    const ragUrl = process.env.RAG_API_URL || 'http://localhost:5000';
    const pythonUrl = process.env.PYTHON_SERVICE_URL || 'http://localhost:8000';

    const [ragHealthy, pythonHealthy] = await Promise.all([
      checkUrlHealthy(`${ragUrl}/status`),
      checkUrlHealthy(`${pythonUrl}/health`),
    ]);

    return NextResponse.json({
      keys: {
        grokConfigured: APIKeyValidator.validateGrokKey(),
        ballDontLieConfigured: Boolean(process.env.BALLDONTLIE_API_KEY),
        stripeConfigured: APIKeyValidator.validateStripeKeys(),
        smtpConfigured: APIKeyValidator.validateSMTPConfig() || process.env.SMTP_HOST === 'localhost',
        ragApiKeyConfigured: Boolean(process.env.FLASK_API_KEY),
      },
      services: {
        rag: { url: ragUrl, healthy: ragHealthy },
        python: { url: pythonUrl, healthy: pythonHealthy },
      },
      env: {
        nodeEnv: process.env.NODE_ENV || 'development',
        baseUrl: process.env.NEXT_PUBLIC_BASE_URL || null,
      },
    });
  } catch (error) {
    console.error('[api/admin/settings] GET failed', error);
    return NextResponse.json({ error: 'Failed to load settings' }, { status: 500 });
  }
}




