/**
 * Admin Performance Monitoring API
 * GET /api/admin/performance - Get system performance metrics and health
 */

import { NextRequest } from 'next/server';
import { AuthMiddleware } from '@/lib/auth';
import { handleApiError } from '@/lib/error-handler';
import { PerformanceDashboard } from '@/lib/performance-middleware';

export async function GET(request: NextRequest) {
  try {
    // Require admin role
    const authResult = await AuthMiddleware.requireRole(request, 'ADMIN');
    if (authResult) {
      return authResult;
    }

    const url = new URL(request.url);
    const timeRange = url.searchParams.get('timeRange') as '1h' | '24h' | '7d' || '24h';

    // Get performance metrics
    const [metrics, health] = await Promise.all([
      PerformanceDashboard.getPerformanceMetrics(timeRange),
      PerformanceDashboard.getSystemHealth(),
    ]);

    return new Response(JSON.stringify({
      success: true,
      data: {
        ...metrics,
        health,
        timestamp: new Date().toISOString(),
      },
    }), {
      status: 200,
      headers: { 'Content-Type': 'application/json' },
    });

  } catch (error) {
    return handleApiError(error as Error, request);
  }
}
