/**
 * Admin Chat Sessions API
 * GET /api/admin/chat/sessions
 * Returns all active chat sessions with messages
 * 
 * PROTECTED: Requires ADMIN role
 */

import { NextRequest, NextResponse } from 'next/server';
import { PrismaClient } from '@prisma/client';
import { AuthMiddleware } from '@/lib/auth';

const prisma = new PrismaClient();

export async function GET(request: NextRequest) {
  try {
    // Require ADMIN role
    const authResponse = await AuthMiddleware.requireRole(request, 'ADMIN');
    if (authResponse) return authResponse;
    // Get all live chat feedback entries from last 7 days
    const sevenDaysAgo = new Date();
    sevenDaysAgo.setDate(sevenDaysAgo.getDate() - 7);

    const feedbackEntries = await prisma.feedback.findMany({
      where: {
        createdAt: { gte: sevenDaysAgo },
      },
      orderBy: { createdAt: 'desc' },
    });

    // Group messages by session ID
    const sessionsMap = new Map<string, any>();

    for (const entry of feedbackEntries) {
      let metadata: any = {};
      try {
        metadata = entry.metadata ? JSON.parse(entry.metadata as string) : {};
      } catch {
        metadata = {};
      }
      const sessionId = metadata.sessionId || entry.id;
      const isLiveChat = metadata.type === 'live_chat';
      const isAdminReply = metadata.type === 'admin_reply';

      if (!sessionsMap.has(sessionId)) {
        sessionsMap.set(sessionId, {
          sessionId,
          userName: metadata.userName || 'Website Visitor',
          userEmail: metadata.userEmail || 'unknown@example.com',
          messages: [],
          lastMessageTime: entry.createdAt,
          unreadCount: 0,
        });
      }

      const session = sessionsMap.get(sessionId);
      
      session.messages.push({
        id: entry.id,
        content: entry.message,
        sender: isAdminReply ? 'admin' : 'user',
        timestamp: entry.createdAt.toISOString(),
      });

      // Update last message time if newer
      if (entry.createdAt > session.lastMessageTime) {
        session.lastMessageTime = entry.createdAt;
      }

      // Count unread (non-admin messages)
      if (!isAdminReply && entry.status === 'OPEN') {
        session.unreadCount++;
      }
    }

    // Convert to array and sort by last message time
    const sessions = Array.from(sessionsMap.values())
      .filter(s => s.messages.length > 0)
      .map(s => ({
        ...s,
        messages: s.messages.sort((a: any, b: any) => 
          new Date(a.timestamp).getTime() - new Date(b.timestamp).getTime()
        ),
        lastMessage: s.messages[s.messages.length - 1]?.content || '',
        lastMessageTime: new Date(s.lastMessageTime).toLocaleString(),
      }))
      .sort((a, b) => 
        new Date(b.lastMessageTime).getTime() - new Date(a.lastMessageTime).getTime()
      );

    return NextResponse.json({ sessions });
  } catch (error) {
    console.error('Failed to fetch chat sessions:', error);
    return NextResponse.json(
      { error: 'Failed to fetch chat sessions' },
      { status: 500 }
    );
  } finally {
    await prisma.$disconnect();
  }
}
