/**
 * Admin Chat Reply API
 * POST /api/admin/chat/reply
 * Sends a reply to a user's chat session
 * 
 * PROTECTED: Requires ADMIN role
 */

import { NextRequest, NextResponse } from 'next/server';
import { z } from 'zod';
import nodemailer from 'nodemailer';
import { PrismaClient } from '@prisma/client';
import { AuthMiddleware } from '@/lib/auth';

const prisma = new PrismaClient();

const replySchema = z.object({
  sessionId: z.string().min(1),
  message: z.string().min(1).max(5000),
});

export async function POST(request: NextRequest) {
  try {
    // Require ADMIN role
    const authResponse = await AuthMiddleware.requireRole(request, 'ADMIN');
    if (authResponse) return authResponse;

    const body = await request.json();
    const { sessionId, message } = replySchema.parse(body);

    // Find the original chat session to get user email
    const originalMessage = await prisma.feedback.findFirst({
      where: {
        OR: [
          { id: sessionId },
          {
            metadata: {
              contains: sessionId,
            },
          },
        ],
      },
      orderBy: { createdAt: 'asc' },
    });

    if (!originalMessage) {
      return NextResponse.json(
        { error: 'Chat session not found' },
        { status: 404 }
      );
    }

    let metadata: any = {};
    try {
      metadata = originalMessage.metadata ? JSON.parse(originalMessage.metadata as string) : {};
    } catch {
      metadata = {};
    }
    const userEmail = metadata.userEmail || 'unknown@example.com';
    const userName = metadata.userName || 'Customer';

    // Store admin reply in database
    await prisma.feedback.create({
      data: {
        userId: 'system', // Use system user for admin replies
        feedbackType: 'other',
        subject: `Admin Reply: ${sessionId}`,
        message: message,
        priority: 'LOW',
        status: 'RESOLVED',
        metadata: JSON.stringify({
          type: 'admin_reply',
          sessionId: sessionId,
          userEmail: userEmail,
          userName: userName,
        }),
      },
    });

    // Mark original messages as resolved
    await prisma.feedback.updateMany({
      where: {
        metadata: {
          contains: sessionId,
        },
      },
      data: {
        status: 'RESOLVED',
      },
    });

    // Send email to user
    try {
      const transporter = nodemailer.createTransport({
        host: process.env.SMTP_HOST || 'localhost',
        port: parseInt(process.env.SMTP_PORT || '25'),
        secure: false,
        tls: { rejectUnauthorized: false },
      });

      const emailHtml = `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; background: #f8fafc; padding: 20px;">
          <div style="background: linear-gradient(135deg, #3b82f6 0%, #8b5cf6 100%); padding: 30px; border-radius: 12px 12px 0 0;">
            <h1 style="color: white; margin: 0; font-size: 24px;">EventheOdds Support</h1>
            <p style="color: rgba(255,255,255,0.9); margin: 10px 0 0 0;">We've responded to your message</p>
          </div>
          
          <div style="background: white; padding: 30px; border-radius: 0 0 12px 12px; border: 1px solid #e2e8f0; border-top: none;">
            <p style="color: #334155; margin: 0 0 20px 0;">Hi ${userName},</p>
            
            <div style="background: #f1f5f9; padding: 20px; border-radius: 8px; border-left: 4px solid #3b82f6;">
              <p style="color: #1e293b; margin: 0; white-space: pre-wrap; line-height: 1.6;">${message}</p>
            </div>

            <p style="color: #64748b; margin: 20px 0; font-size: 14px;">
              You can reply directly to this email if you have any follow-up questions.
            </p>

            <hr style="border: none; border-top: 1px solid #e2e8f0; margin: 20px 0;" />

            <p style="color: #94a3b8; font-size: 12px; margin: 0;">
              This is a response to your support request on EventheOdds.ai<br />
              Reference: ${sessionId}
            </p>
          </div>
        </div>
      `;

      await transporter.sendMail({
        from: `"EventheOdds Support" <${process.env.EMAIL_FROM || 'support@eventheodds.ai'}>`,
        to: userEmail,
        replyTo: process.env.SUPPORT_EMAIL || 'admin@eventheodds.ai',
        subject: `Re: Your EventheOdds Support Request`,
        html: emailHtml,
        text: `Hi ${userName},\n\n${message}\n\nYou can reply directly to this email if you have any follow-up questions.\n\nReference: ${sessionId}\n\n-- EventheOdds Support Team`,
      });

      console.log(`Admin reply sent to ${userEmail} for session ${sessionId}`);
    } catch (emailError) {
      console.error('Failed to send reply email:', emailError);
      // Don't fail the request - message is saved in DB
    }

    return NextResponse.json({
      success: true,
      message: 'Reply sent successfully',
    });

  } catch (error) {
    console.error('Admin chat reply error:', error);

    if (error instanceof z.ZodError) {
      return NextResponse.json(
        { error: 'Invalid request', details: error.issues },
        { status: 400 }
      );
    }

    return NextResponse.json(
      { error: 'Failed to send reply' },
      { status: 500 }
    );
  } finally {
    await prisma.$disconnect();
  }
}
