import { NextRequest, NextResponse } from 'next/server';
import { AuthMiddleware } from '@/lib/auth';
import { prisma } from '@/lib/database';

function safeJsonParse<T>(value: any, fallback: T): T {
  if (!value || typeof value !== 'string') return fallback;
  try {
    return JSON.parse(value) as T;
  } catch {
    return fallback;
  }
}

function extractSummary(resultsJson: any, tradesJson: any) {
  const resultsObj = resultsJson?.results || resultsJson || {};
  const totalTrades =
    resultsJson?.total_trades ??
    resultsObj?.total_trades ??
    (Array.isArray(tradesJson) ? tradesJson.length : 0);

  const rawWinRate =
    resultsObj?.win_rate ??
    resultsJson?.win_rate ??
    null;
  const winRate = rawWinRate == null ? null : Number(rawWinRate);

  const totalProfit =
    resultsObj?.total_profit ??
    resultsJson?.total_profit ??
    null;

  return {
    totalTrades: Number(totalTrades || 0),
    winRate,
    totalProfit: totalProfit == null ? null : Number(totalProfit),
  };
}

export async function GET(request: NextRequest) {
  try {
    const authResponse = await AuthMiddleware.requireRole(request, 'ADMIN');
    if (authResponse) return authResponse;

    const url = new URL(request.url);
    const limit = Math.min(200, Math.max(1, Number(url.searchParams.get('limit') || 50)));

    const rows = await prisma.backtestResult.findMany({
      orderBy: { createdAt: 'desc' },
      take: limit,
    });

    const userIds = Array.from(new Set(rows.map((r) => r.userId).filter(Boolean))) as string[];
    const users = userIds.length
      ? await prisma.user.findMany({
          where: { id: { in: userIds } },
          select: { id: true, email: true, firstName: true, lastName: true },
        })
      : [];
    const userMap = new Map(users.map((u) => [u.id, u]));

    const backtests = rows.map((bt) => {
      const results = safeJsonParse<any>(bt.results, {});
      const trades = safeJsonParse<any[]>(bt.trades, []);
      const summary = extractSummary(results, trades);
      const user = bt.userId ? userMap.get(bt.userId) : null;

      return {
        id: bt.id,
        jobId: bt.jobId,
        userId: bt.userId,
        userEmail: user?.email || null,
        userName: user ? `${user.firstName || ''} ${user.lastName || ''}`.trim() : null,
        market: bt.market,
        strategyName: bt.strategyName,
        status: bt.status,
        executionTime: bt.executionTime,
        createdAt: bt.createdAt,
        summary,
      };
    });

    return NextResponse.json({ backtests, total: backtests.length });
  } catch (error) {
    console.error('[api/admin/backtests] GET failed', error);
    return NextResponse.json({ error: 'Failed to load backtests' }, { status: 500 });
  }
}




