/**
 * Admin Announcements API
 * GET /api/admin/announcements - List all announcements (including unpublished)
 * POST /api/admin/announcements - Create a new announcement
 * 
 * PROTECTED: Requires ADMIN role
 */

import { NextRequest, NextResponse } from 'next/server';
import { z } from 'zod';
import { prisma } from '@/lib/database';
import { AuthMiddleware } from '@/lib/auth';

const announcementSchema = z.object({
  title: z.string().min(1, 'Title is required').max(200),
  content: z.string().min(1, 'Content is required').max(5000),
  type: z.enum(['feature', 'update', 'security', 'performance', 'maintenance']),
  important: z.boolean().default(false),
  published: z.boolean().default(true),
  date: z.string().optional(), // ISO date string
});

export async function GET(request: NextRequest) {
  try {
    // Require ADMIN role
    const authResponse = await AuthMiddleware.requireRole(request, 'ADMIN');
    if (authResponse) return authResponse;
    const rows = await prisma.systemAnalytics.findMany({
      where: { eventType: 'ANNOUNCEMENT' },
      orderBy: { createdAt: 'desc' },
      take: 100,
    });

    const announcements = rows.map((row) => {
      let event: any = {};
      try {
        event = JSON.parse(row.eventData || '{}');
      } catch {}

      return {
        id: row.id,
        title: event.title || 'Untitled',
        content: event.content || '',
        type: event.type || 'update',
        important: !!event.important,
        published: event.published !== false,
        date: event.date || row.createdAt.toISOString().split('T')[0],
        createdAt: row.createdAt.toISOString(),
      };
    });

    return NextResponse.json({ announcements });
  } catch (error) {
    console.error('Failed to fetch announcements:', error);
    return NextResponse.json(
      { error: 'Failed to fetch announcements' },
      { status: 500 }
    );
  }
}

export async function POST(request: NextRequest) {
  try {
    // Require ADMIN role
    const authResponse = await AuthMiddleware.requireRole(request, 'ADMIN');
    if (authResponse) return authResponse;

    const body = await request.json();
    const data = announcementSchema.parse(body);

    const announcement = await prisma.systemAnalytics.create({
      data: {
        eventType: 'ANNOUNCEMENT',
        eventData: JSON.stringify({
          title: data.title,
          content: data.content,
          type: data.type,
          important: data.important,
          published: data.published,
          date: data.date || new Date().toISOString().split('T')[0],
        }),
        metadata: JSON.stringify({ createdBy: 'admin' }),
      },
    });

    return NextResponse.json({
      success: true,
      id: announcement.id,
      message: 'Announcement created successfully',
    });
  } catch (error: any) {
    console.error('Failed to create announcement:', error);
    console.error('Error message:', error?.message);
    console.error('Error stack:', error?.stack);

    if (error instanceof z.ZodError) {
      return NextResponse.json(
        { error: 'Validation failed', details: error.issues },
        { status: 400 }
      );
    }

    return NextResponse.json(
      { error: 'Failed to create announcement', details: error?.message || 'Unknown error' },
      { status: 500 }
    );
  }
}
