import { NextRequest, NextResponse } from 'next/server';
import { AuthMiddleware } from '@/lib/auth';
import { prisma } from '@/lib/database';
import { PerformanceDashboard } from '@/lib/performance-middleware';

function toISODate(d: Date): string {
  return d.toISOString().slice(0, 10);
}

export async function GET(request: NextRequest) {
  try {
    const authResponse = await AuthMiddleware.requireRole(request, 'ADMIN');
    if (authResponse) return authResponse;

    const now = new Date();
    const since7d = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000);
    const since30d = new Date(now.getTime() - 30 * 24 * 60 * 60 * 1000);

    const [totalUsers, activeUsers, totalStrategies, totalRuns] = await Promise.all([
      prisma.user.count(),
      prisma.user.count({ where: { lastLoginAt: { gte: since7d } } }),
      prisma.strategy.count(),
      prisma.backtestResult.count(),
    ]);

    const perf = await PerformanceDashboard.getPerformanceMetrics('24h');
    const totalPerfTime = (perf.apiResponseTimes || []).reduce((sum: number, m: any) => sum + Number(m?.totalTime || 0), 0);
    const totalPerfCount = (perf.apiResponseTimes || []).reduce((sum: number, m: any) => sum + Number(m?.count || 0), 0);
    const avgResponseTime = totalPerfCount > 0 ? Math.round(totalPerfTime / totalPerfCount) : 0;
    const errorRate = Number(perf?.errorRates?.errorRate || 0);

    // Market usage (based on last 30 days of backtests)
    const marketGroups = await prisma.backtestResult.groupBy({
      by: ['market'],
      where: { createdAt: { gte: since30d } },
      _count: { _all: true },
    });
    const marketTotal = marketGroups.reduce((sum, g) => sum + g._count._all, 0);
    const marketUsage: Record<string, number> = {};
    if (marketTotal > 0) {
      marketGroups
        .sort((a, b) => b._count._all - a._count._all)
        .slice(0, 10)
        .forEach((g) => {
          marketUsage[g.market] = Math.round((g._count._all / marketTotal) * 100);
        });
    } else {
      marketUsage.sports = 100;
    }

    // Daily backtests (last 7 days)
    const runs = await prisma.backtestResult.findMany({
      where: { createdAt: { gte: since7d } },
      select: { createdAt: true },
      orderBy: { createdAt: 'asc' },
      take: 20000,
    });

    const counts: Record<string, number> = {};
    for (const r of runs) {
      const key = toISODate(r.createdAt);
      counts[key] = (counts[key] || 0) + 1;
    }

    const dailyRuns: { date: string; count: number }[] = [];
    for (let i = 6; i >= 0; i--) {
      const d = new Date(now.getTime() - i * 24 * 60 * 60 * 1000);
      const key = toISODate(d);
      dailyRuns.push({ date: key, count: counts[key] || 0 });
    }

    return NextResponse.json({
      totalUsers,
      activeUsers,
      totalStrategies,
      totalRuns,
      avgResponseTime,
      errorRate,
      marketUsage,
      dailyRuns,
    });
  } catch (error) {
    console.error('[api/admin/analytics] GET failed', error);
    return NextResponse.json({ error: 'Failed to load analytics' }, { status: 500 });
  }
}




