import { NextRequest, NextResponse } from 'next/server';
import { cookies } from 'next/headers';
import jwt from 'jsonwebtoken';
import { DatabaseService } from '@/lib/database';
import fs from 'fs/promises';
import path from 'path';

// AI Guru Settings - stored in a JSON file (can be migrated to DB later)
const SETTINGS_FILE = path.join(process.cwd(), 'data', 'ai-guru-settings.json');

interface AIGuruSettings {
  // LLM Settings
  temperature: number;           // 0.0 - 1.0 (creativity)
  maxTokens: number;             // Max response length
  
  // RAG Settings
  ragEnabled: boolean;           // Toggle RAG on/off
  ragRelevanceThreshold: number; // 0.0 - 1.0 (default 0.6)
  useEnrichedChunks: boolean;    // Use enriched vs raw chunks
  maxContextLength: number;      // Max chars for RAG context
  ragTopK: number;               // Number of chunks to retrieve
  
  // System Prompt
  systemPromptPrefix: string;    // Custom prefix to system prompt
  systemPromptSuffix: string;    // Custom suffix to system prompt
  
  // Behavior Settings
  includeThemesInResponse: boolean;   // Show topic themes
  includeKeyPointsInResponse: boolean; // Show key takeaways
  includeSourcesInResponse: boolean;   // Show source references
  
  // Rate Limiting (per user)
  maxQueriesPerMinute: number;
  maxQueriesPerDay: number;
  
  // Last updated
  updatedAt: string;
  updatedBy: string;
}

const DEFAULT_SETTINGS: AIGuruSettings = {
  temperature: 0.7,
  maxTokens: 2000,
  ragEnabled: true,
  ragRelevanceThreshold: 0.6,
  useEnrichedChunks: true,
  maxContextLength: 3000,
  ragTopK: 5,
  systemPromptPrefix: '',
  systemPromptSuffix: '',
  includeThemesInResponse: true,
  includeKeyPointsInResponse: true,
  includeSourcesInResponse: true,
  maxQueriesPerMinute: 10,
  maxQueriesPerDay: 100,
  updatedAt: new Date().toISOString(),
  updatedBy: 'system',
};

// Verify admin access
async function verifyAdminAccess(request: NextRequest): Promise<{ authorized: boolean; userId?: string; email?: string }> {
  const cookieStore = await cookies();
  const authToken = cookieStore.get('auth-token')?.value;

  if (!authToken) {
    return { authorized: false };
  }

  try {
    const jwtSecret = process.env.JWT_SECRET;
    if (!jwtSecret) {
      return { authorized: false };
    }

    const decoded = jwt.verify(authToken, jwtSecret) as {
      userId: string;
      email: string;
    };

    const user = await DatabaseService.findUserById(decoded.userId);
    
    if (!user || user.role !== 'ADMIN' || !user.emailVerified) {
      return { authorized: false };
    }

    return { authorized: true, userId: user.id, email: user.email };
  } catch {
    return { authorized: false };
  }
}

// Load settings from file
async function loadSettings(): Promise<AIGuruSettings> {
  try {
    const data = await fs.readFile(SETTINGS_FILE, 'utf-8');
    return { ...DEFAULT_SETTINGS, ...JSON.parse(data) };
  } catch {
    // File doesn't exist, return defaults
    return DEFAULT_SETTINGS;
  }
}

// Save settings to file
async function saveSettings(settings: AIGuruSettings): Promise<void> {
  // Ensure data directory exists
  const dir = path.dirname(SETTINGS_FILE);
  await fs.mkdir(dir, { recursive: true });
  await fs.writeFile(SETTINGS_FILE, JSON.stringify(settings, null, 2));
}

// GET - Retrieve current settings
export async function GET(request: NextRequest) {
  const { authorized } = await verifyAdminAccess(request);
  
  if (!authorized) {
    return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
  }

  try {
    const settings = await loadSettings();
    
    // Also get RAG status
    let ragStatus = { healthy: false, documents: 0, enriched: 0 };
    try {
      const ragUrl = process.env.RAG_API_URL || 'http://localhost:5001';
      const ragRes = await fetch(`${ragUrl}/documents`, { 
        headers: { 'X-API-Key': process.env.FLASK_API_KEY || '' },
        signal: AbortSignal.timeout(5000)
      });
      if (ragRes.ok) {
        const ragData = await ragRes.json();
        ragStatus = {
          healthy: true,
          documents: ragData.total_chunks || 0,
          enriched: ragData.total_enriched || 0,
        };
      }
    } catch {
      // RAG not available
    }

    return NextResponse.json({
      settings,
      ragStatus,
      defaults: DEFAULT_SETTINGS,
    });
  } catch (error) {
    console.error('Error loading AI Guru settings:', error);
    return NextResponse.json({ error: 'Failed to load settings' }, { status: 500 });
  }
}

// PUT - Update settings
export async function PUT(request: NextRequest) {
  const { authorized, email } = await verifyAdminAccess(request);
  
  if (!authorized) {
    return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
  }

  try {
    const body = await request.json();
    const currentSettings = await loadSettings();
    
    // Validate and merge settings
    const newSettings: AIGuruSettings = {
      temperature: Math.max(0, Math.min(1, body.temperature ?? currentSettings.temperature)),
      maxTokens: Math.max(100, Math.min(4000, body.maxTokens ?? currentSettings.maxTokens)),
      ragEnabled: body.ragEnabled ?? currentSettings.ragEnabled,
      ragRelevanceThreshold: Math.max(0, Math.min(1, body.ragRelevanceThreshold ?? currentSettings.ragRelevanceThreshold)),
      useEnrichedChunks: body.useEnrichedChunks ?? currentSettings.useEnrichedChunks,
      maxContextLength: Math.max(500, Math.min(10000, body.maxContextLength ?? currentSettings.maxContextLength)),
      ragTopK: Math.max(1, Math.min(20, body.ragTopK ?? currentSettings.ragTopK)),
      systemPromptPrefix: (body.systemPromptPrefix ?? currentSettings.systemPromptPrefix).slice(0, 2000),
      systemPromptSuffix: (body.systemPromptSuffix ?? currentSettings.systemPromptSuffix).slice(0, 2000),
      includeThemesInResponse: body.includeThemesInResponse ?? currentSettings.includeThemesInResponse,
      includeKeyPointsInResponse: body.includeKeyPointsInResponse ?? currentSettings.includeKeyPointsInResponse,
      includeSourcesInResponse: body.includeSourcesInResponse ?? currentSettings.includeSourcesInResponse,
      maxQueriesPerMinute: Math.max(1, Math.min(100, body.maxQueriesPerMinute ?? currentSettings.maxQueriesPerMinute)),
      maxQueriesPerDay: Math.max(10, Math.min(10000, body.maxQueriesPerDay ?? currentSettings.maxQueriesPerDay)),
      updatedAt: new Date().toISOString(),
      updatedBy: email || 'admin',
    };

    await saveSettings(newSettings);

    return NextResponse.json({
      success: true,
      settings: newSettings,
    });
  } catch (error) {
    console.error('Error saving AI Guru settings:', error);
    return NextResponse.json({ error: 'Failed to save settings' }, { status: 500 });
  }
}

// POST - Reset to defaults
export async function POST(request: NextRequest) {
  const { authorized, email } = await verifyAdminAccess(request);
  
  if (!authorized) {
    return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
  }

  try {
    const body = await request.json();
    
    if (body.action === 'reset') {
      const resetSettings: AIGuruSettings = {
        ...DEFAULT_SETTINGS,
        updatedAt: new Date().toISOString(),
        updatedBy: email || 'admin',
      };
      
      await saveSettings(resetSettings);
      
      return NextResponse.json({
        success: true,
        settings: resetSettings,
        message: 'Settings reset to defaults',
      });
    }

    return NextResponse.json({ error: 'Unknown action' }, { status: 400 });
  } catch (error) {
    console.error('Error resetting AI Guru settings:', error);
    return NextResponse.json({ error: 'Failed to reset settings' }, { status: 500 });
  }
}

