/**
 * Authentication API Integration Tests
 * Tests signup, login, and authentication flows end-to-end
 */

import { NextRequest } from 'next/server';
import { POST as signupHandler } from '../auth/signup/route';
import { POST as loginHandler } from '../auth/login/route';
import { POST as verifyEmailHandler } from '../auth/verify-email/route';
import { DatabaseService } from '@/lib/database';
import { EmailUtils } from '@/lib/email';

jest.mock('@/lib/database');
jest.mock('@/lib/email');

describe('Authentication API Integration Tests', () => {
  beforeEach(() => {
    jest.clearAllMocks();
  });

  describe('POST /api/auth/signup', () => {
    it('should create new user successfully', async () => {
      const mockUser = {
        id: 'user_123',
        email: 'test@example.com',
        firstName: 'John',
        lastName: 'Doe',
        subscriptionTier: 'FREE',
        emailVerificationToken: 'verification-token-123',
      };

      (DatabaseService.createUser as jest.Mock).mockResolvedValue(mockUser);
      (EmailUtils.sendEmailVerification as jest.Mock).mockResolvedValue(true);

      const request = new NextRequest('http://localhost:3000/api/auth/signup', {
        method: 'POST',
        body: JSON.stringify({
          email: 'test@example.com',
          password: 'password123',
          firstName: 'John',
          lastName: 'Doe',
          subscriptionTier: 'FREE',
        }),
      });

      const response = await signupHandler(request);
      const data = await response.json();

      expect(response.status).toBe(201);
      expect(data.success).toBe(true);
      expect(data.user.email).toBe('test@example.com');
      expect(data.token).toBeDefined();

      expect(DatabaseService.createUser).toHaveBeenCalledWith({
        email: 'test@example.com',
        password: 'password123',
        firstName: 'John',
        lastName: 'Doe',
        subscriptionTier: 'FREE',
        emailVerificationToken: expect.any(String),
        emailVerificationExpires: expect.any(Date),
      });

      expect(EmailUtils.sendEmailVerification).toHaveBeenCalled();
    });

    it('should validate required fields', async () => {
      const request = new NextRequest('http://localhost:3000/api/auth/signup', {
        method: 'POST',
        body: JSON.stringify({
          email: 'test@example.com',
          // Missing password and names
        }),
      });

      const response = await signupHandler(request);
      const data = await response.json();

      expect(response.status).toBe(400);
      expect(data.error).toContain('Validation failed');
    });

    it('should reject duplicate emails', async () => {
      (DatabaseService.findUserByEmail as jest.Mock).mockResolvedValue({
        id: 'existing_user',
        email: 'test@example.com',
      });

      const request = new NextRequest('http://localhost:3000/api/auth/signup', {
        method: 'POST',
        body: JSON.stringify({
          email: 'test@example.com',
          password: 'password123',
          firstName: 'John',
          lastName: 'Doe',
        }),
      });

      const response = await signupHandler(request);
      const data = await response.json();

      expect(response.status).toBe(409);
      expect(data.error).toContain('already exists');
    });

    it('should handle email sending failures gracefully', async () => {
      const mockUser = {
        id: 'user_123',
        email: 'test@example.com',
        firstName: 'John',
        lastName: 'Doe',
        subscriptionTier: 'FREE',
      };

      (DatabaseService.createUser as jest.Mock).mockResolvedValue(mockUser);
      (EmailUtils.sendEmailVerification as jest.Mock).mockResolvedValue(false);

      const request = new NextRequest('http://localhost:3000/api/auth/signup', {
        method: 'POST',
        body: JSON.stringify({
          email: 'test@example.com',
          password: 'password123',
          firstName: 'John',
          lastName: 'Doe',
        }),
      });

      const response = await signupHandler(request);
      const data = await response.json();

      // Should still succeed even if email fails
      expect(response.status).toBe(201);
      expect(data.success).toBe(true);
      expect(data.user.id).toBe('user_123');
    });
  });

  describe('POST /api/auth/login', () => {
    it('should authenticate valid user', async () => {
      const mockUser = {
        id: 'user_123',
        email: 'test@example.com',
        passwordHash: 'hashed_password',
        subscriptionTier: 'PRO',
        emailVerified: true,
      };

      (DatabaseService.findUserByEmail as jest.Mock).mockResolvedValue(mockUser);
      (DatabaseService.verifyPassword as jest.Mock).mockResolvedValue(true);
      (DatabaseService.updateUserLastLogin as jest.Mock).mockResolvedValue(undefined);
      (DatabaseService.recordUserEvent as jest.Mock).mockResolvedValue(undefined);

      const request = new NextRequest('http://localhost:3000/api/auth/login', {
        method: 'POST',
        body: JSON.stringify({
          email: 'test@example.com',
          password: 'password123',
        }),
      });

      const response = await loginHandler(request);
      const data = await response.json();

      expect(response.status).toBe(200);
      expect(data.success).toBe(true);
      expect(data.user.email).toBe('test@example.com');
      expect(data.token).toBeDefined();

      expect(DatabaseService.findUserByEmail).toHaveBeenCalledWith('test@example.com');
      expect(DatabaseService.verifyPassword).toHaveBeenCalledWith('password123', 'hashed_password');
      expect(DatabaseService.recordUserEvent).toHaveBeenCalled();
    });

    it('should reject invalid credentials', async () => {
      (DatabaseService.findUserByEmail as jest.Mock).mockResolvedValue({
        id: 'user_123',
        email: 'test@example.com',
      });
      (DatabaseService.verifyPassword as jest.Mock).mockResolvedValue(false);

      const request = new NextRequest('http://localhost:3000/api/auth/login', {
        method: 'POST',
        body: JSON.stringify({
          email: 'test@example.com',
          password: 'wrongpassword',
        }),
      });

      const response = await loginHandler(request);
      const data = await response.json();

      expect(response.status).toBe(401);
      expect(data.error).toContain('Invalid email or password');
    });

    it('should reject unverified emails', async () => {
      const mockUser = {
        id: 'user_123',
        email: 'test@example.com',
        subscriptionTier: 'FREE',
        emailVerified: false,
      };

      (DatabaseService.findUserByEmail as jest.Mock).mockResolvedValue(mockUser);
      (DatabaseService.verifyPassword as jest.Mock).mockResolvedValue(true);

      const request = new NextRequest('http://localhost:3000/api/auth/login', {
        method: 'POST',
        body: JSON.stringify({
          email: 'test@example.com',
          password: 'password123',
        }),
      });

      const response = await loginHandler(request);
      const data = await response.json();

      expect(response.status).toBe(403);
      expect(data.error).toContain('Email not verified');
    });

    it('should implement rate limiting', async () => {
      // Mock multiple failed attempts
      (DatabaseService.findUserByEmail as jest.Mock).mockResolvedValue({
        id: 'user_123',
        email: 'test@example.com',
      });
      (DatabaseService.verifyPassword as jest.Mock).mockResolvedValue(false);

      // Make multiple requests to trigger rate limiting
      for (let i = 0; i < 6; i++) {
        const request = new NextRequest('http://localhost:3000/api/auth/login', {
          method: 'POST',
          body: JSON.stringify({
            email: 'test@example.com',
            password: 'wrongpassword',
          }),
        });
        await loginHandler(request);
      }

      // The rate limiting is implemented in the middleware, so this test verifies
      // that the login handler doesn't crash under repeated failed attempts
      expect(DatabaseService.findUserByEmail).toHaveBeenCalled();
    });
  });

  describe('POST /api/auth/verify-email', () => {
    it('should verify email with valid token', async () => {
      const mockUser = {
        id: 'user_123',
        email: 'test@example.com',
        emailVerificationExpires: new Date(Date.now() + 3600000), // 1 hour from now
      };

      (DatabaseService.findUserByVerificationToken as jest.Mock).mockResolvedValue(mockUser);
      (DatabaseService.updateUser as jest.Mock).mockResolvedValue(undefined);
      (DatabaseService.recordUserEvent as jest.Mock).mockResolvedValue(undefined);

      const request = new NextRequest('http://localhost:3000/api/auth/verify-email', {
        method: 'POST',
        body: JSON.stringify({
          token: 'valid-verification-token',
        }),
      });

      const response = await verifyEmailHandler(request);
      const data = await response.json();

      expect(response.status).toBe(200);
      expect(data.success).toBe(true);
      expect(data.message).toContain('verified successfully');

      expect(DatabaseService.updateUser).toHaveBeenCalledWith('user_123', {
        emailVerified: true,
        emailVerifiedAt: expect.any(Date),
        emailVerificationToken: null,
        emailVerificationExpires: null,
      });
    });

    it('should reject expired tokens', async () => {
      const mockUser = {
        id: 'user_123',
        email: 'test@example.com',
        emailVerificationExpires: new Date(Date.now() - 3600000), // 1 hour ago
      };

      (DatabaseService.findUserByVerificationToken as jest.Mock).mockResolvedValue(mockUser);

      const request = new NextRequest('http://localhost:3000/api/auth/verify-email', {
        method: 'POST',
        body: JSON.stringify({
          token: 'expired-token',
        }),
      });

      const response = await verifyEmailHandler(request);
      const data = await response.json();

      expect(response.status).toBe(400);
      expect(data.error).toContain('expired');
    });

    it('should reject invalid tokens', async () => {
      (DatabaseService.findUserByVerificationToken as jest.Mock).mockResolvedValue(null);

      const request = new NextRequest('http://localhost:3000/api/auth/verify-email', {
        method: 'POST',
        body: JSON.stringify({
          token: 'invalid-token',
        }),
      });

      const response = await verifyEmailHandler(request);
      const data = await response.json();

      expect(response.status).toBe(400);
      expect(data.error).toContain('Invalid or expired');
    });
  });
});
