import { ArrowUpRight, Database, Zap, Activity } from 'lucide-react';

import { DataService } from '@/services/DataService';
import { prisma } from '@/lib/database';
import { PerformanceDashboard } from '@/lib/performance-middleware';

export const dynamic = 'force-dynamic';

export default async function AdminDashboard() {
  const dataService = DataService.getInstance();
  const datasets = await dataService.getAllDatasetsMeta();

  const totalRows = datasets.reduce((acc, d) => acc + d.rowCount, 0);
  const [totalBacktests, totalStrategies, perf, health] = await Promise.all([
    prisma.backtestResult.count(),
    prisma.strategy.count(),
    PerformanceDashboard.getPerformanceMetrics('24h'),
    PerformanceDashboard.getSystemHealth(),
  ]);

  const totalPerfTime = (perf.apiResponseTimes || []).reduce((sum: number, m: any) => sum + Number(m?.totalTime || 0), 0);
  const totalPerfCount = (perf.apiResponseTimes || []).reduce((sum: number, m: any) => sum + Number(m?.count || 0), 0);
  const avgResponseTime = totalPerfCount > 0 ? Math.round(totalPerfTime / totalPerfCount) : 0;

  return (
    <div className="space-y-8">
      <div>
        <h1 className="text-2xl font-bold text-white">Dashboard Overview</h1>
        <p className="text-slate-400">System health and performance metrics.</p>
      </div>

      <div className="grid gap-6 md:grid-cols-3">
        <StatCard
          title="Total Data Points"
          value={totalRows.toLocaleString()}
          icon={Database}
          trend="Live"
        />
        <StatCard
          title="Backtests Run"
          value={totalBacktests.toLocaleString()}
          icon={Zap}
          trend="All-time"
        />
        <StatCard
          title="Total Strategies"
          value={totalStrategies.toString()}
          icon={Activity}
          trend="All-time"
        />
      </div>

      <div className="grid gap-6 md:grid-cols-2">
        <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
          <h3 className="font-semibold text-white">Recent Ingestion</h3>
          <div className="mt-4 space-y-4">
            {datasets.map((ds) => (
              <div
                key={ds.id}
                className="flex items-center justify-between rounded-lg bg-slate-950/50 p-3"
              >
                <div className="flex items-center gap-3">
                  <div className="flex h-8 w-8 items-center justify-center rounded bg-blue-500/20 text-blue-400">
                    <Database className="h-4 w-4" />
                  </div>
                  <div>
                    <p className="text-sm font-medium text-slate-200">
                      {ds.domain.toUpperCase()} Master
                    </p>
                    <p className="text-xs text-slate-500">
                      Updated {new Date(ds.lastModified).toLocaleTimeString()}
                    </p>
                  </div>
                </div>
                <div className="text-right">
                  <p className="text-sm font-medium text-white">
                    {ds.rowCount} rows
                  </p>
                  <p className="text-xs text-slate-500">
                    {(ds.sizeBytes / 1024).toFixed(1)} KB
                  </p>
                </div>
              </div>
            ))}
          </div>
        </div>

        <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
          <h3 className="font-semibold text-white">System Health</h3>
          <div className="mt-4 space-y-4">
            <div className="flex items-center justify-between border-b border-white/5 pb-4 last:border-0">
              <div>
                <p className="text-sm font-medium text-slate-200">API Latency</p>
                <p className="text-xs text-slate-500">Average (last 24h)</p>
              </div>
              <span className="text-sm font-bold text-emerald-400">{avgResponseTime}ms</span>
            </div>
            <div className="flex items-center justify-between border-b border-white/5 pb-4 last:border-0">
              <div>
                <p className="text-sm font-medium text-slate-200">
                  Database
                </p>
                <p className="text-xs text-slate-500">Health check</p>
              </div>
              {health.checks.database?.status === 'pass' ? (
                <span className="flex items-center gap-2 text-sm font-bold text-emerald-400">
                  <span className="h-2 w-2 rounded-full bg-emerald-400" />
                  Healthy
                </span>
              ) : (
                <span className="flex items-center gap-2 text-sm font-bold text-rose-400">
                  <span className="h-2 w-2 rounded-full bg-rose-400" />
                  Failing
                </span>
              )}
            </div>
            <div className="flex items-center justify-between border-b border-white/5 pb-4 last:border-0">
              <div>
                <p className="text-sm font-medium text-slate-200">
                  Cache
                </p>
                <p className="text-xs text-slate-500">Health check</p>
              </div>
              {health.checks.cache?.status === 'pass' ? (
                <span className="flex items-center gap-2 text-sm font-bold text-emerald-400">
                  <span className="h-2 w-2 rounded-full bg-emerald-400" />
                  Healthy
                </span>
              ) : (
                <span className="flex items-center gap-2 text-sm font-bold text-rose-400">
                  <span className="h-2 w-2 rounded-full bg-rose-400" />
                  Failing
                </span>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}

function StatCard({
  title,
  value,
  icon: Icon,
  trend,
}: {
  title: string;
  value: string;
  icon: any;
  trend: string;
}) {
  return (
    <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6 transition hover:border-blue-500/30">
      <div className="flex items-start justify-between">
        <div>
          <p className="text-sm font-medium text-slate-400">{title}</p>
          <h3 className="mt-2 text-3xl font-bold text-white">{value}</h3>
        </div>
        <div className="rounded-lg bg-white/5 p-2 text-slate-300">
          <Icon className="h-6 w-6" />
        </div>
      </div>
      <div className="mt-4 flex items-center gap-2 text-xs text-emerald-400">
        <ArrowUpRight className="h-3 w-3" />
        <span>{trend}</span>
      </div>
    </div>
  );
}

