import { AdminHeader } from '@/components/admin/AdminHeader';
import { AdminSidebar } from '@/components/admin/AdminSidebar';
import { redirect } from 'next/navigation';
import { cookies } from 'next/headers';
import jwt from 'jsonwebtoken';
import { DatabaseService } from '@/lib/database';

// Server-side auth verification - runs BEFORE any rendering
async function verifyAdminAccess(): Promise<{ authorized: boolean; userId?: string }> {
  const cookieStore = await cookies();
  const authToken = cookieStore.get('auth-token')?.value;

  if (!authToken) {
    return { authorized: false };
  }

  try {
    const jwtSecret = process.env.JWT_SECRET;
    if (!jwtSecret) {
      console.error('JWT_SECRET not configured');
      return { authorized: false };
    }

    // Verify JWT token
    const decoded = jwt.verify(authToken, jwtSecret) as {
      userId: string;
      email: string;
      subscriptionTier: string;
      role?: string;
    };

    // Fetch user from database to verify they still exist and check role
    const user = await DatabaseService.findUserById(decoded.userId);
    
    if (!user) {
      return { authorized: false };
    }

    // Admin area is restricted to ADMIN role
    const hasAdminAccess = user.role === 'ADMIN';

    if (!hasAdminAccess) {
      return { authorized: false };
    }

    // Verify email is verified
    if (!user.emailVerified) {
      return { authorized: false };
    }

    return { authorized: true, userId: user.id };
  } catch (error) {
    console.error('Admin auth verification failed:', error);
    return { authorized: false };
  }
}

export default async function AdminLayout({
  children,
}: {
  children: React.ReactNode;
}) {
  // CRITICAL: Verify auth BEFORE rendering ANY content
  const { authorized } = await verifyAdminAccess();

  // Redirect immediately if not authorized - no content leak
  if (!authorized) {
    redirect('/login');
  }

  return (
    <div className="flex h-screen bg-slate-950 font-sans text-slate-100 antialiased selection:bg-blue-500/30">
      <AdminSidebar />
      <div className="flex flex-1 flex-col overflow-hidden">
        <AdminHeader />
        <main className="flex-1 overflow-y-auto bg-slate-950 p-8">
          <div className="mx-auto max-w-6xl">{children}</div>
        </main>
      </div>
    </div>
  );
}

