'use client';

import React, { useState, useEffect, useRef } from 'react';
import Link from 'next/link';
import { ArrowLeft, Send, User, Users, RefreshCw, Clock, Mail, MessageSquare } from 'lucide-react';

interface ChatSession {
  sessionId: string;
  userName: string;
  userEmail: string;
  lastMessage: string;
  lastMessageTime: string;
  unreadCount: number;
  messages: ChatMessage[];
}

interface ChatMessage {
  id: string;
  content: string;
  sender: 'user' | 'admin';
  timestamp: string;
}

export default function AdminChatPage() {
  const [sessions, setSessions] = useState<ChatSession[]>([]);
  const [selectedSession, setSelectedSession] = useState<string | null>(null);
  const [replyMessage, setReplyMessage] = useState('');
  const [isLoading, setIsLoading] = useState(true);
  const [isSending, setIsSending] = useState(false);
  const [lastRefresh, setLastRefresh] = useState<Date>(new Date());
  const messagesEndRef = useRef<HTMLDivElement>(null);

  // Fetch chat sessions
  const fetchSessions = async () => {
    try {
      const response = await fetch('/api/admin/chat/sessions', {
        credentials: 'include',
      });
      if (response.ok) {
        const data = await response.json();
        setSessions(data.sessions || []);
      }
    } catch (error) {
      console.error('Failed to fetch sessions:', error);
    } finally {
      setIsLoading(false);
      setLastRefresh(new Date());
    }
  };

  // Auto-refresh every 10 seconds
  useEffect(() => {
    fetchSessions();
    const interval = setInterval(fetchSessions, 10000);
    return () => clearInterval(interval);
  }, []);

  // Scroll to bottom when messages change
  useEffect(() => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  }, [selectedSession, sessions]);

  const handleSendReply = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!replyMessage.trim() || !selectedSession || isSending) return;

    setIsSending(true);
    try {
      const response = await fetch('/api/admin/chat/reply', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'include',
        body: JSON.stringify({
          sessionId: selectedSession,
          message: replyMessage,
        }),
      });

      if (response.ok) {
        setReplyMessage('');
        fetchSessions(); // Refresh to show new message
      } else {
        alert('Failed to send reply');
      }
    } catch (error) {
      console.error('Failed to send reply:', error);
      alert('Failed to send reply');
    } finally {
      setIsSending(false);
    }
  };

  const selectedChat = sessions.find(s => s.sessionId === selectedSession);

  return (
    <div className="min-h-screen bg-slate-950">
      {/* Header */}
      <header className="bg-slate-900/50 border-b border-white/10">
        <div className="mx-auto max-w-7xl px-6 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-4">
              <Link href="/admin" className="text-slate-400 hover:text-white transition">
                <ArrowLeft className="h-5 w-5" />
              </Link>
              <div>
                <h1 className="text-2xl font-bold text-white">Live Chat Support</h1>
                <p className="text-slate-400 text-sm">Respond to customer inquiries</p>
              </div>
            </div>
            <div className="flex items-center gap-4">
              <span className="text-slate-400 text-sm flex items-center gap-2">
                <Clock className="h-4 w-4" />
                Last updated: {lastRefresh.toLocaleTimeString()}
              </span>
              <button
                onClick={fetchSessions}
                className="flex items-center gap-2 bg-slate-800 hover:bg-slate-700 text-white px-4 py-2 rounded-lg transition"
              >
                <RefreshCw className="h-4 w-4" />
                Refresh
              </button>
            </div>
          </div>
        </div>
      </header>

      <div className="mx-auto max-w-7xl px-6 py-6">
        <div className="grid grid-cols-12 gap-6 h-[calc(100vh-180px)]">
          {/* Sessions List */}
          <div className="col-span-4 bg-slate-900/50 border border-white/10 rounded-xl overflow-hidden flex flex-col">
            <div className="p-4 border-b border-white/10">
              <h2 className="text-lg font-semibold text-white flex items-center gap-2">
                <Users className="h-5 w-5 text-blue-400" />
                Active Conversations
                {sessions.length > 0 && (
                  <span className="bg-blue-500 text-white text-xs px-2 py-0.5 rounded-full">
                    {sessions.length}
                  </span>
                )}
              </h2>
            </div>

            <div className="flex-1 overflow-y-auto">
              {isLoading ? (
                <div className="p-8 text-center text-slate-400">
                  <RefreshCw className="h-8 w-8 animate-spin mx-auto mb-2" />
                  Loading conversations...
                </div>
              ) : sessions.length === 0 ? (
                <div className="p-8 text-center text-slate-400">
                  <MessageSquare className="h-12 w-12 mx-auto mb-3 opacity-50" />
                  <p>No active conversations</p>
                  <p className="text-sm mt-1">New chats will appear here</p>
                </div>
              ) : (
                sessions.map((session) => (
                  <button
                    key={session.sessionId}
                    onClick={() => setSelectedSession(session.sessionId)}
                    className={`w-full p-4 text-left border-b border-white/5 hover:bg-slate-800/50 transition ${
                      selectedSession === session.sessionId ? 'bg-slate-800/50 border-l-2 border-l-blue-500' : ''
                    }`}
                  >
                    <div className="flex items-start justify-between">
                      <div className="flex items-center gap-3">
                        <div className="w-10 h-10 bg-blue-500/20 rounded-full flex items-center justify-center">
                          <User className="h-5 w-5 text-blue-400" />
                        </div>
                        <div>
                          <p className="font-semibold text-white">{session.userName || 'Anonymous'}</p>
                          <p className="text-slate-400 text-sm">{session.userEmail}</p>
                        </div>
                      </div>
                      {session.unreadCount > 0 && (
                        <span className="bg-red-500 text-white text-xs px-2 py-0.5 rounded-full">
                          {session.unreadCount}
                        </span>
                      )}
                    </div>
                    <p className="text-slate-400 text-sm mt-2 truncate">{session.lastMessage}</p>
                    <p className="text-slate-500 text-xs mt-1">{session.lastMessageTime}</p>
                  </button>
                ))
              )}
            </div>
          </div>

          {/* Chat Window */}
          <div className="col-span-8 bg-slate-900/50 border border-white/10 rounded-xl overflow-hidden flex flex-col">
            {selectedChat ? (
              <>
                {/* Chat Header */}
                <div className="p-4 border-b border-white/10 bg-slate-800/50">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-3">
                      <div className="w-10 h-10 bg-blue-500/20 rounded-full flex items-center justify-center">
                        <User className="h-5 w-5 text-blue-400" />
                      </div>
                      <div>
                        <p className="font-semibold text-white">{selectedChat.userName || 'Anonymous'}</p>
                        <p className="text-slate-400 text-sm flex items-center gap-1">
                          <Mail className="h-3 w-3" />
                          {selectedChat.userEmail}
                        </p>
                      </div>
                    </div>
                    <a
                      href={`mailto:${selectedChat.userEmail}?subject=Re: Your EventheOdds Support Request`}
                      className="bg-slate-700 hover:bg-slate-600 text-white px-4 py-2 rounded-lg text-sm transition flex items-center gap-2"
                    >
                      <Mail className="h-4 w-4" />
                      Send Email
                    </a>
                  </div>
                </div>

                {/* Messages */}
                <div className="flex-1 overflow-y-auto p-4 space-y-4">
                  {selectedChat.messages.map((message) => (
                    <div
                      key={message.id}
                      className={`flex gap-3 ${message.sender === 'admin' ? 'flex-row-reverse' : ''}`}
                    >
                      <div className={`w-8 h-8 rounded-full flex items-center justify-center flex-shrink-0 ${
                        message.sender === 'admin' 
                          ? 'bg-green-500' 
                          : 'bg-blue-500'
                      }`}>
                        {message.sender === 'admin' ? (
                          <Users className="h-4 w-4 text-white" />
                        ) : (
                          <User className="h-4 w-4 text-white" />
                        )}
                      </div>
                      <div className={`max-w-[70%] ${message.sender === 'admin' ? 'text-right' : ''}`}>
                        <div className={`rounded-2xl px-4 py-2 ${
                          message.sender === 'admin'
                            ? 'bg-green-600 text-white rounded-tr-sm'
                            : 'bg-slate-800 text-slate-200 rounded-tl-sm'
                        }`}>
                          <p className="text-sm whitespace-pre-wrap">{message.content}</p>
                        </div>
                        <p className="text-xs text-slate-500 mt-1">
                          {new Date(message.timestamp).toLocaleString()}
                        </p>
                      </div>
                    </div>
                  ))}
                  <div ref={messagesEndRef} />
                </div>

                {/* Reply Input */}
                <form onSubmit={handleSendReply} className="p-4 border-t border-white/10">
                  <div className="flex gap-2">
                    <textarea
                      value={replyMessage}
                      onChange={(e) => setReplyMessage(e.target.value)}
                      onKeyDown={(e) => {
                        if (e.key === 'Enter' && !e.shiftKey) {
                          e.preventDefault();
                          handleSendReply(e);
                        }
                      }}
                      placeholder="Type your reply... (Enter to send, Shift+Enter for new line)"
                      disabled={isSending}
                      rows={2}
                      className="flex-1 bg-slate-800 border border-slate-600 rounded-lg px-4 py-2 text-white placeholder-slate-400 focus:border-green-500 focus:outline-none disabled:opacity-50 resize-none"
                    />
                    <button
                      type="submit"
                      disabled={!replyMessage.trim() || isSending}
                      className="px-6 bg-green-600 hover:bg-green-500 disabled:opacity-50 disabled:cursor-not-allowed text-white rounded-lg font-semibold transition flex items-center gap-2"
                    >
                      <Send className="h-4 w-4" />
                      Send
                    </button>
                  </div>
                  <p className="text-xs text-slate-500 mt-2">
                    Your reply will be sent to the user via email and shown in their chat
                  </p>
                </form>
              </>
            ) : (
              <div className="flex-1 flex items-center justify-center text-slate-400">
                <div className="text-center">
                  <MessageSquare className="h-16 w-16 mx-auto mb-4 opacity-50" />
                  <p className="text-lg">Select a conversation to start chatting</p>
                  <p className="text-sm mt-1">Choose from the list on the left</p>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}
