'use client';

import { useEffect, useState, useCallback } from 'react';
import { Bot, Brain, Database, MessageSquare, Save, RefreshCw, Zap, Shield, Sliders } from 'lucide-react';

interface AIGuruSettings {
  temperature: number;
  maxTokens: number;
  ragEnabled: boolean;
  ragRelevanceThreshold: number;
  useEnrichedChunks: boolean;
  maxContextLength: number;
  ragTopK: number;
  systemPromptPrefix: string;
  systemPromptSuffix: string;
  includeThemesInResponse: boolean;
  includeKeyPointsInResponse: boolean;
  includeSourcesInResponse: boolean;
  maxQueriesPerMinute: number;
  maxQueriesPerDay: number;
  updatedAt: string;
  updatedBy: string;
}

interface RAGStatus {
  healthy: boolean;
  documents: number;
  enriched: number;
}

export default function AIGuruSettingsPage() {
  const [settings, setSettings] = useState<AIGuruSettings | null>(null);
  const [defaults, setDefaults] = useState<AIGuruSettings | null>(null);
  const [ragStatus, setRagStatus] = useState<RAGStatus | null>(null);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState<string | null>(null);

  const loadSettings = useCallback(async () => {
    try {
      setLoading(true);
      setError(null);
      
      const token = localStorage.getItem('authToken');
      const headers: Record<string, string> = { 'Content-Type': 'application/json' };
      if (token) headers.Authorization = `Bearer ${token}`;

      const res = await fetch('/api/admin/ai-guru', { headers, credentials: 'include' });
      
      if (!res.ok) {
        const body = await res.json().catch(() => ({}));
        throw new Error(body?.error || `Failed to load settings (${res.status})`);
      }

      const data = await res.json();
      setSettings(data.settings);
      setDefaults(data.defaults);
      setRagStatus(data.ragStatus);
    } catch (e) {
      setError(e instanceof Error ? e.message : 'Failed to load settings');
    } finally {
      setLoading(false);
    }
  }, []);

  useEffect(() => {
    void loadSettings();
  }, [loadSettings]);

  const saveSettings = async () => {
    if (!settings) return;
    
    try {
      setSaving(true);
      setError(null);
      setSuccess(null);
      
      const token = localStorage.getItem('authToken');
      const headers: Record<string, string> = { 'Content-Type': 'application/json' };
      if (token) headers.Authorization = `Bearer ${token}`;

      const res = await fetch('/api/admin/ai-guru', {
        method: 'PUT',
        headers,
        credentials: 'include',
        body: JSON.stringify(settings),
      });

      if (!res.ok) {
        const body = await res.json().catch(() => ({}));
        throw new Error(body?.error || 'Failed to save settings');
      }

      const data = await res.json();
      setSettings(data.settings);
      setSuccess('Settings saved successfully!');
      setTimeout(() => setSuccess(null), 3000);
    } catch (e) {
      setError(e instanceof Error ? e.message : 'Failed to save settings');
    } finally {
      setSaving(false);
    }
  };

  const resetToDefaults = async () => {
    if (!confirm('Reset all AI Guru settings to defaults?')) return;
    
    try {
      setSaving(true);
      setError(null);
      
      const token = localStorage.getItem('authToken');
      const headers: Record<string, string> = { 'Content-Type': 'application/json' };
      if (token) headers.Authorization = `Bearer ${token}`;

      const res = await fetch('/api/admin/ai-guru', {
        method: 'POST',
        headers,
        credentials: 'include',
        body: JSON.stringify({ action: 'reset' }),
      });

      if (!res.ok) throw new Error('Failed to reset settings');

      const data = await res.json();
      setSettings(data.settings);
      setSuccess('Settings reset to defaults!');
      setTimeout(() => setSuccess(null), 3000);
    } catch (e) {
      setError(e instanceof Error ? e.message : 'Failed to reset');
    } finally {
      setSaving(false);
    }
  };

  const updateSetting = <K extends keyof AIGuruSettings>(key: K, value: AIGuruSettings[K]) => {
    setSettings(prev => prev ? { ...prev, [key]: value } : null);
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  return (
    <div className="space-y-8">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold text-white flex items-center gap-3">
            <Bot className="h-7 w-7 text-amber-400" />
            AI Sport Guru Settings
          </h1>
          <p className="text-slate-400 mt-1">
            Configure the AI assistant&apos;s behavior, RAG integration, and response settings.
          </p>
        </div>
        <div className="flex gap-3">
          <button
            onClick={resetToDefaults}
            disabled={saving}
            className="flex items-center gap-2 px-4 py-2 bg-slate-800 hover:bg-slate-700 text-slate-300 rounded-lg transition-colors disabled:opacity-50"
          >
            <RefreshCw className="h-4 w-4" />
            Reset Defaults
          </button>
          <button
            onClick={saveSettings}
            disabled={saving}
            className="flex items-center gap-2 px-4 py-2 bg-blue-600 hover:bg-blue-500 text-white rounded-lg transition-colors disabled:opacity-50"
          >
            <Save className="h-4 w-4" />
            {saving ? 'Saving...' : 'Save Changes'}
          </button>
        </div>
      </div>

      {/* Alerts */}
      {error && (
        <div className="rounded-xl border border-red-500/20 bg-red-500/10 p-4 text-red-200">
          {error}
        </div>
      )}
      {success && (
        <div className="rounded-xl border border-green-500/20 bg-green-500/10 p-4 text-green-200">
          {success}
        </div>
      )}

      {/* RAG Status Card */}
      <div className="rounded-2xl border border-white/10 bg-gradient-to-br from-amber-500/10 to-orange-500/10 p-6">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-3">
            <Database className="h-6 w-6 text-amber-400" />
            <div>
              <h3 className="font-semibold text-white">RAG Knowledge Base</h3>
              <p className="text-sm text-slate-400">Document retrieval system status</p>
            </div>
          </div>
          <div className="flex items-center gap-6">
            <div className="text-center">
              <div className="text-2xl font-bold text-white">{ragStatus?.documents || 0}</div>
              <div className="text-xs text-slate-400">Total Chunks</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-emerald-400">{ragStatus?.enriched || 0}</div>
              <div className="text-xs text-slate-400">Enriched</div>
            </div>
            <div className={`px-3 py-1 rounded-full text-sm font-medium ${ragStatus?.healthy ? 'bg-emerald-500/20 text-emerald-400' : 'bg-red-500/20 text-red-400'}`}>
              {ragStatus?.healthy ? '● Online' : '● Offline'}
            </div>
          </div>
        </div>
      </div>

      {settings && (
        <div className="grid gap-6 lg:grid-cols-2">
          {/* LLM Settings */}
          <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
            <div className="flex items-center gap-2 mb-4">
              <Brain className="h-5 w-5 text-purple-400" />
              <h3 className="font-semibold text-white">LLM Response Settings</h3>
            </div>
            <div className="space-y-5">
              <div>
                <div className="flex justify-between mb-2">
                  <label className="text-sm text-slate-300">Temperature (Creativity)</label>
                  <span className="text-sm text-amber-400 font-mono">{settings.temperature.toFixed(2)}</span>
                </div>
                <input
                  type="range"
                  min="0"
                  max="1"
                  step="0.05"
                  value={settings.temperature}
                  onChange={(e) => updateSetting('temperature', parseFloat(e.target.value))}
                  className="w-full accent-amber-400"
                />
                <div className="flex justify-between text-xs text-slate-500 mt-1">
                  <span>Precise (0.0)</span>
                  <span>Creative (1.0)</span>
                </div>
              </div>
              
              <div>
                <div className="flex justify-between mb-2">
                  <label className="text-sm text-slate-300">Max Response Tokens</label>
                  <span className="text-sm text-amber-400 font-mono">{settings.maxTokens}</span>
                </div>
                <input
                  type="range"
                  min="500"
                  max="4000"
                  step="100"
                  value={settings.maxTokens}
                  onChange={(e) => updateSetting('maxTokens', parseInt(e.target.value))}
                  className="w-full accent-amber-400"
                />
              </div>
            </div>
          </div>

          {/* RAG Settings */}
          <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
            <div className="flex items-center gap-2 mb-4">
              <Zap className="h-5 w-5 text-amber-400" />
              <h3 className="font-semibold text-white">RAG Integration</h3>
            </div>
            <div className="space-y-5">
              <div className="flex items-center justify-between">
                <label className="text-sm text-slate-300">Enable RAG Knowledge</label>
                <button
                  onClick={() => updateSetting('ragEnabled', !settings.ragEnabled)}
                  className={`relative w-12 h-6 rounded-full transition-colors ${settings.ragEnabled ? 'bg-emerald-500' : 'bg-slate-700'}`}
                >
                  <div className={`absolute top-0.5 w-5 h-5 rounded-full bg-white transition-transform ${settings.ragEnabled ? 'translate-x-6' : 'translate-x-0.5'}`} />
                </button>
              </div>
              
              <div className="flex items-center justify-between">
                <label className="text-sm text-slate-300">Use Enriched Chunks</label>
                <button
                  onClick={() => updateSetting('useEnrichedChunks', !settings.useEnrichedChunks)}
                  className={`relative w-12 h-6 rounded-full transition-colors ${settings.useEnrichedChunks ? 'bg-emerald-500' : 'bg-slate-700'}`}
                >
                  <div className={`absolute top-0.5 w-5 h-5 rounded-full bg-white transition-transform ${settings.useEnrichedChunks ? 'translate-x-6' : 'translate-x-0.5'}`} />
                </button>
              </div>

              <div>
                <div className="flex justify-between mb-2">
                  <label className="text-sm text-slate-300">Relevance Threshold</label>
                  <span className="text-sm text-amber-400 font-mono">{settings.ragRelevanceThreshold.toFixed(2)}</span>
                </div>
                <input
                  type="range"
                  min="0.3"
                  max="0.95"
                  step="0.05"
                  value={settings.ragRelevanceThreshold}
                  onChange={(e) => updateSetting('ragRelevanceThreshold', parseFloat(e.target.value))}
                  className="w-full accent-amber-400"
                />
                <div className="flex justify-between text-xs text-slate-500 mt-1">
                  <span>More results</span>
                  <span>Higher quality</span>
                </div>
              </div>

              <div>
                <div className="flex justify-between mb-2">
                  <label className="text-sm text-slate-300">Top K Results</label>
                  <span className="text-sm text-amber-400 font-mono">{settings.ragTopK}</span>
                </div>
                <input
                  type="range"
                  min="1"
                  max="15"
                  step="1"
                  value={settings.ragTopK}
                  onChange={(e) => updateSetting('ragTopK', parseInt(e.target.value))}
                  className="w-full accent-amber-400"
                />
              </div>

              <div>
                <div className="flex justify-between mb-2">
                  <label className="text-sm text-slate-300">Max Context Length</label>
                  <span className="text-sm text-amber-400 font-mono">{settings.maxContextLength}</span>
                </div>
                <input
                  type="range"
                  min="1000"
                  max="8000"
                  step="500"
                  value={settings.maxContextLength}
                  onChange={(e) => updateSetting('maxContextLength', parseInt(e.target.value))}
                  className="w-full accent-amber-400"
                />
              </div>
            </div>
          </div>

          {/* Response Display Settings */}
          <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
            <div className="flex items-center gap-2 mb-4">
              <MessageSquare className="h-5 w-5 text-blue-400" />
              <h3 className="font-semibold text-white">Response Display</h3>
            </div>
            <div className="space-y-4">
              <div className="flex items-center justify-between p-3 rounded-lg bg-slate-950/50">
                <label className="text-sm text-slate-300">Show Topic Themes</label>
                <button
                  onClick={() => updateSetting('includeThemesInResponse', !settings.includeThemesInResponse)}
                  className={`relative w-12 h-6 rounded-full transition-colors ${settings.includeThemesInResponse ? 'bg-emerald-500' : 'bg-slate-700'}`}
                >
                  <div className={`absolute top-0.5 w-5 h-5 rounded-full bg-white transition-transform ${settings.includeThemesInResponse ? 'translate-x-6' : 'translate-x-0.5'}`} />
                </button>
              </div>
              <div className="flex items-center justify-between p-3 rounded-lg bg-slate-950/50">
                <label className="text-sm text-slate-300">Show Key Points</label>
                <button
                  onClick={() => updateSetting('includeKeyPointsInResponse', !settings.includeKeyPointsInResponse)}
                  className={`relative w-12 h-6 rounded-full transition-colors ${settings.includeKeyPointsInResponse ? 'bg-emerald-500' : 'bg-slate-700'}`}
                >
                  <div className={`absolute top-0.5 w-5 h-5 rounded-full bg-white transition-transform ${settings.includeKeyPointsInResponse ? 'translate-x-6' : 'translate-x-0.5'}`} />
                </button>
              </div>
              <div className="flex items-center justify-between p-3 rounded-lg bg-slate-950/50">
                <label className="text-sm text-slate-300">Show Source References</label>
                <button
                  onClick={() => updateSetting('includeSourcesInResponse', !settings.includeSourcesInResponse)}
                  className={`relative w-12 h-6 rounded-full transition-colors ${settings.includeSourcesInResponse ? 'bg-emerald-500' : 'bg-slate-700'}`}
                >
                  <div className={`absolute top-0.5 w-5 h-5 rounded-full bg-white transition-transform ${settings.includeSourcesInResponse ? 'translate-x-6' : 'translate-x-0.5'}`} />
                </button>
              </div>
            </div>
          </div>

          {/* Rate Limiting */}
          <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
            <div className="flex items-center gap-2 mb-4">
              <Shield className="h-5 w-5 text-rose-400" />
              <h3 className="font-semibold text-white">Rate Limiting</h3>
            </div>
            <div className="space-y-5">
              <div>
                <div className="flex justify-between mb-2">
                  <label className="text-sm text-slate-300">Queries Per Minute</label>
                  <span className="text-sm text-amber-400 font-mono">{settings.maxQueriesPerMinute}</span>
                </div>
                <input
                  type="range"
                  min="1"
                  max="30"
                  step="1"
                  value={settings.maxQueriesPerMinute}
                  onChange={(e) => updateSetting('maxQueriesPerMinute', parseInt(e.target.value))}
                  className="w-full accent-amber-400"
                />
              </div>
              <div>
                <div className="flex justify-between mb-2">
                  <label className="text-sm text-slate-300">Queries Per Day</label>
                  <span className="text-sm text-amber-400 font-mono">{settings.maxQueriesPerDay}</span>
                </div>
                <input
                  type="range"
                  min="10"
                  max="500"
                  step="10"
                  value={settings.maxQueriesPerDay}
                  onChange={(e) => updateSetting('maxQueriesPerDay', parseInt(e.target.value))}
                  className="w-full accent-amber-400"
                />
              </div>
            </div>
          </div>

          {/* System Prompt Customization */}
          <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6 lg:col-span-2">
            <div className="flex items-center gap-2 mb-4">
              <Sliders className="h-5 w-5 text-cyan-400" />
              <h3 className="font-semibold text-white">System Prompt Customization</h3>
            </div>
            <div className="grid gap-6 lg:grid-cols-2">
              <div>
                <label className="block text-sm text-slate-300 mb-2">Prompt Prefix (added before default prompt)</label>
                <textarea
                  value={settings.systemPromptPrefix}
                  onChange={(e) => updateSetting('systemPromptPrefix', e.target.value)}
                  placeholder="e.g., Always be conservative with risk estimates..."
                  className="w-full h-32 px-4 py-3 rounded-lg bg-slate-950 border border-white/10 text-white placeholder-slate-500 resize-none focus:outline-none focus:ring-2 focus:ring-amber-400/50"
                />
              </div>
              <div>
                <label className="block text-sm text-slate-300 mb-2">Prompt Suffix (added after default prompt)</label>
                <textarea
                  value={settings.systemPromptSuffix}
                  onChange={(e) => updateSetting('systemPromptSuffix', e.target.value)}
                  placeholder="e.g., End each response with a risk disclaimer..."
                  className="w-full h-32 px-4 py-3 rounded-lg bg-slate-950 border border-white/10 text-white placeholder-slate-500 resize-none focus:outline-none focus:ring-2 focus:ring-amber-400/50"
                />
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Last Updated Info */}
      {settings && (
        <div className="text-center text-sm text-slate-500">
          Last updated: {new Date(settings.updatedAt).toLocaleString()} by {settings.updatedBy}
        </div>
      )}
    </div>
  );
}

