#!/usr/bin/env python3
"""Test the data as a user would experience in chat."""

import json
from pathlib import Path

DATA_DIR = Path("/var/www/html/eventheodds/data")

print("=" * 70)
print("TESTING USER CHAT EXPERIENCE")
print("=" * 70)

# Test 1: NBA standings
print()
print("User Query: Show me NBA standings for 2025 season")
print()
print("AI Response:")
print("-" * 50)

nba_standings = DATA_DIR / "nba/standings.json"
if nba_standings.exists():
    with open(nba_standings) as f:
        standings = json.load(f)
    
    print("**NBA Standings 2025 Season** (BallDontLie API)")
    print()
    
    standings.sort(key=lambda x: x.get("wins", 0), reverse=True)
    
    for i, team in enumerate(standings[:10], 1):
        name = team.get("team", {}).get("full_name", "Unknown")
        wins = team.get("wins", 0)
        losses = team.get("losses", 0)
        conf = team.get("conference", "N/A")
        pct = wins / (wins + losses) if (wins + losses) > 0 else 0
        print(f"  {i:2}. {name}: {wins}-{losses} (.{int(pct*1000):03d}) - {conf}")
else:
    print("  No NBA standings data found")

# Test 2: NFL odds
print()
print("-" * 50)
print()
print("User Query: Show betting odds for NFL games")
print()
print("AI Response:")
print("-" * 50)

nfl_odds = DATA_DIR / "odds/nfl_odds.json"
if nfl_odds.exists():
    with open(nfl_odds) as f:
        odds = json.load(f)
    
    vendors = list(set(o.get("vendor") for o in odds if o.get("vendor")))
    games = list(set(o.get("game_id") for o in odds if o.get("game_id")))
    
    print("**NFL Betting Odds** (Real Sportsbook Data)")
    print(f"- Games covered: {len(games)}")
    vendor_str = ", ".join(vendors[:4])
    print(f"- Sportsbooks: {len(vendors)} ({vendor_str}...)")
    print()
    
    # Sample odds
    for o in odds[:3]:
        spread = o.get("spread_home_value", "N/A")
        ml_home = o.get("moneyline_home_odds", "N/A")
        total = o.get("total_value", "N/A")
        vendor = o.get("vendor", "")
        print(f"  {vendor}: Spread {spread}, ML {ml_home}, O/U {total}")
else:
    print("  No NFL odds data found")

# Test 3: Player props
print()
print("-" * 50)
print()
print("User Query: Backtest player prop over strategy NFL 2025")
print()
print("AI Response:")
print("-" * 50)

props = DATA_DIR / "player_props/nfl_player_props.json"
if props.exists():
    with open(props) as f:
        data = json.load(f)
    
    prop_types = {}
    for p in data:
        pt = p.get("prop_type", "unknown")
        prop_types[pt] = prop_types.get(pt, 0) + 1
    
    print("**NFL Player Props** (Live Sportsbook Data)")
    print(f"- Total props: {len(data):,}")
    print("- Prop types available:")
    for pt, count in sorted(prop_types.items(), key=lambda x: -x[1])[:8]:
        print(f"    - {pt}: {count} props")
else:
    print("  No player props data found")

# Test 4: NHL data
print()
print("-" * 50)
print()
print("User Query: Show me NHL standings and recent games")
print()
print("AI Response:")
print("-" * 50)

nhl_standings = DATA_DIR / "nhl/standings.json"
if nhl_standings.exists():
    with open(nhl_standings) as f:
        standings = json.load(f)
    print(f"**NHL Standings** - {len(standings)} teams")
    if standings:
        sample = standings[0]
        print(f"  Sample: {sample.get('team', {}).get('full_name', 'Unknown')}")
else:
    print("  No NHL standings")

nhl_games = DATA_DIR / "nhl/games.json"
if nhl_games.exists():
    with open(nhl_games) as f:
        games = json.load(f)
    print(f"**NHL Games** - {len(games)} games on record")
else:
    print("  No NHL games")

# Summary
print()
print("=" * 70)
print("DATA AVAILABILITY SUMMARY")
print("=" * 70)
print()

total_entries = 0
total_files = 0

for sport_dir in sorted(DATA_DIR.iterdir()):
    if not sport_dir.is_dir() or sport_dir.name.startswith('.'):
        continue
    if sport_dir.name in ['cache', 'charts', 'csv', 'balldontlie_cache', 'sportsdata_cache']:
        continue
    
    sport_entries = 0
    sport_files = 0
    
    for f in sport_dir.glob('*.json'):
        if 'metadata' in f.name or 'summary' in f.name or 'index' in f.name:
            continue
        try:
            with open(f) as file:
                data = json.load(file)
                count = len(data) if isinstance(data, list) else 1
                sport_entries += count
                sport_files += 1
        except:
            pass
    
    if sport_entries > 0:
        print(f"  {sport_dir.name.upper()}: {sport_entries:,} entries ({sport_files} files)")
        total_entries += sport_entries
        total_files += sport_files

print()
print(f"TOTAL: {total_entries:,} entries in {total_files} files")
print()
print("✅ All data is ready for user queries!")
