#!/bin/bash
# Production Setup Script for EvenTheOdds.ai
# Run with: sudo bash scripts/setup-production.sh

set -e

echo "🚀 EvenTheOdds.ai Production Setup Script"
echo "=========================================="

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Check if running as root
if [ "$EUID" -ne 0 ]; then
    echo -e "${RED}Please run as root (sudo bash scripts/setup-production.sh)${NC}"
    exit 1
fi

# 1. Enable required Apache modules
echo -e "${YELLOW}1. Enabling Apache modules...${NC}"
a2enmod proxy proxy_http proxy_wstunnel rewrite headers ssl
echo -e "${GREEN}✓ Apache modules enabled${NC}"

# 2. Copy Apache vhost configs
echo -e "${YELLOW}2. Setting up Apache virtual hosts...${NC}"
cp /var/www/html/leadgen/backtest/config/apache/eventheodds.conf /etc/apache2/sites-available/
cp /var/www/html/leadgen/backtest/config/apache/eventheodds-ssl.conf /etc/apache2/sites-available/
a2ensite eventheodds.conf
echo -e "${GREEN}✓ Virtual hosts configured${NC}"

# 3. Test Apache config
echo -e "${YELLOW}3. Testing Apache configuration...${NC}"
apache2ctl configtest
echo -e "${GREEN}✓ Apache configuration valid${NC}"

# 4. Install certbot if not present
echo -e "${YELLOW}4. Checking certbot installation...${NC}"
if ! command -v certbot &> /dev/null; then
    apt update
    apt install -y certbot python3-certbot-apache
    echo -e "${GREEN}✓ Certbot installed${NC}"
else
    echo -e "${GREEN}✓ Certbot already installed${NC}"
fi

# 5. Obtain SSL certificates
echo -e "${YELLOW}5. Obtaining SSL certificates...${NC}"
echo -e "${YELLOW}   Make sure DNS is pointing to this server first!${NC}"
read -p "Are DNS records for eventheodds.ai and eventheoddsai.com pointing to this server? (y/n): " dns_ready

if [ "$dns_ready" = "y" ]; then
    certbot --apache -d eventheodds.ai -d www.eventheodds.ai -d eventheoddsai.com -d www.eventheoddsai.com
    
    # Enable SSL vhost after certificates are obtained
    a2ensite eventheodds-ssl.conf
    echo -e "${GREEN}✓ SSL certificates obtained and enabled${NC}"
else
    echo -e "${YELLOW}Skipping SSL setup. Run 'sudo certbot --apache -d eventheodds.ai -d www.eventheodds.ai -d eventheoddsai.com -d www.eventheoddsai.com' when DNS is ready.${NC}"
fi

# 6. Reload Apache
echo -e "${YELLOW}6. Reloading Apache...${NC}"
systemctl reload apache2
echo -e "${GREEN}✓ Apache reloaded${NC}"

# 7. Setup certbot auto-renewal
echo -e "${YELLOW}7. Setting up SSL auto-renewal...${NC}"
systemctl enable certbot.timer
systemctl start certbot.timer
echo -e "${GREEN}✓ Certbot auto-renewal enabled${NC}"

# 8. Create .env template if not exists
echo -e "${YELLOW}8. Checking environment configuration...${NC}"
ENV_FILE="/var/www/html/leadgen/backtest/.env"
if [ ! -f "$ENV_FILE" ]; then
    cat > "$ENV_FILE" << 'ENVEOF'
# Database
DATABASE_URL="mysql://user:password@localhost:3306/backtest_db"

# Authentication
JWT_SECRET="your-very-long-and-secure-jwt-secret-minimum-32-characters"

# AI API (xAI Grok)
GROK_API_KEY="xai-your-api-key-here"

# Stripe
STRIPE_PUBLISHABLE_KEY="pk_live_..."
STRIPE_SECRET_KEY="sk_live_..."
STRIPE_WEBHOOK_SECRET="whsec_..."

# Email (SMTP)
SMTP_HOST="smtp.your-provider.com"
SMTP_PORT="587"
SMTP_USER="your-email@domain.com"
SMTP_PASS="your-email-password"
SMTP_FROM="noreply@eventheodds.ai"

# Python Microservice
PYTHON_SERVICE_URL="http://localhost:8000"
FLASK_API_KEY="your-flask-api-key"

# CORS
ALLOWED_ORIGINS="https://eventheodds.ai,https://www.eventheodds.ai,https://eventheoddsai.com,https://www.eventheoddsai.com"

# Production Mode
NODE_ENV="production"
NEXT_PUBLIC_BASE_URL="https://eventheodds.ai/backtest"
ENVEOF
    chmod 600 "$ENV_FILE"
    echo -e "${RED}⚠️  .env file created with template values. EDIT IT NOW with real credentials!${NC}"
else
    echo -e "${GREEN}✓ .env file exists${NC}"
fi

# 9. Setup systemd services
echo -e "${YELLOW}9. Setting up systemd services...${NC}"
mkdir -p /var/log/backtest
chown www-data:www-data /var/log/backtest

# Copy service files
cp /var/www/html/leadgen/backtest/config/systemd/backtest-python.service /etc/systemd/system/
cp /var/www/html/leadgen/backtest/config/systemd/backtest-nextjs.service /etc/systemd/system/

# Reload systemd
systemctl daemon-reload
systemctl enable backtest-python.service
systemctl enable backtest-nextjs.service
echo -e "${GREEN}✓ Systemd services configured${NC}"

echo ""
echo "=========================================="
echo -e "${GREEN}✅ Production setup complete!${NC}"
echo ""
echo "Next steps:"
echo "1. Edit /var/www/html/leadgen/backtest/.env with real credentials"
echo "2. Rebuild Next.js: cd /var/www/html/leadgen/backtest && npm run build"
echo "3. Start services:"
echo "   sudo systemctl start backtest-python"
echo "   sudo systemctl start backtest-nextjs"
echo ""
echo "Or use PM2 instead:"
echo "   pm2 start npm --name 'backtest' -- start"
echo "   pm2 start 'python -m uvicorn main:app --host 127.0.0.1 --port 8000' --name 'backtest-python' --cwd /var/www/html/leadgen/backtest/python_service"
echo ""
echo "Test the setup:"
echo "  curl -I https://eventheodds.ai/backtest"
echo "  curl http://localhost:8000/health"
echo "=========================================="
