#!/usr/bin/env node

/**
 * E2E Test Runner with Enhanced Console Output
 * Runs Playwright tests with detailed logging and debugging
 */

const { execSync, spawn } = require('child_process');
const fs = require('fs');
const path = require('path');

class E2ETestRunner {
  constructor() {
    this.testResults = {
      total: 0,
      passed: 0,
      failed: 0,
      skipped: 0,
      duration: 0,
      tests: []
    };
    this.startTime = Date.now();
  }

  log(message, level = 'info') {
    const timestamp = new Date().toISOString();
    const prefix = {
      info: 'ℹ️ ',
      success: '✅',
      error: '❌',
      warning: '⚠️ ',
      debug: '🔍'
    }[level] || '📝';

    console.log(`[${timestamp}] ${prefix} ${message}`);
  }

  async checkPrerequisites() {
    this.log('Checking test prerequisites...', 'info');

    // Check if .env exists
    const envPath = path.join(process.cwd(), '.env');
    if (!fs.existsSync(envPath)) {
      this.log('Warning: .env file not found - tests may fail without proper configuration', 'warning');
    } else {
      this.log('✅ .env file found', 'success');
    }

    // Check if database is configured
    const envContent = fs.existsSync(envPath) ? fs.readFileSync(envPath, 'utf8') : '';
    const hasDbUrl = envContent.includes('DATABASE_URL=');
    const hasStripeKeys = envContent.includes('STRIPE_');

    if (hasDbUrl) {
      this.log('✅ Database configuration found', 'success');
    } else {
      this.log('⚠️  Database URL not configured - database tests will fail', 'warning');
    }

    if (hasStripeKeys) {
      this.log('✅ Stripe configuration found', 'success');
    } else {
      this.log('⚠️  Stripe keys not configured - payment tests will fail', 'warning');
    }

    // Check if test user credentials exist
    const hasTestUser = envContent.includes('TEST_USER_EMAIL=');
    if (hasTestUser) {
      this.log('✅ Test user credentials configured', 'success');
    } else {
      this.log('ℹ️  Test user credentials not found - using default test accounts', 'info');
    }
  }

  async setupTestEnvironment() {
    this.log('Setting up test environment...', 'info');

    try {
      // Ensure test database is ready (if using separate test DB)
      if (process.env.NODE_ENV === 'test') {
        this.log('Running in test environment - checking database connection', 'info');

        // You could add database migration/reset logic here
        // execSync('npm run db:reset:test', { stdio: 'inherit' });
      }

      // Check if dev server is running
      try {
        const response = await fetch('http://localhost:3000');
        if (response.ok) {
          this.log('✅ Dev server is running', 'success');
        }
      } catch (error) {
        this.log('⚠️  Dev server not detected - tests will start it automatically', 'warning');
      }

    } catch (error) {
      this.log(`Error setting up test environment: ${error.message}`, 'error');
    }
  }

  async runTests(options = {}) {
    const {
      headed = false,
      debug = false,
      browser = 'chromium',
      grep = '',
      timeout = 60000
    } = options;

    this.log(`Starting E2E tests with Playwright...`, 'info');
    this.log(`Configuration: headed=${headed}, debug=${debug}, browser=${browser}, timeout=${timeout}ms`, 'debug');

    const playwrightArgs = [
      'test',
      `--browser=${browser}`,
      `--timeout=${timeout}`,
      '--reporter=line,json:test-results.json,html:test-results'
    ];

    if (headed) {
      playwrightArgs.push('--headed');
    }

    if (debug) {
      playwrightArgs.push('--debug');
    }

    if (grep) {
      playwrightArgs.push(`--grep=${grep}`);
    }

    // Add environment variables
    const env = {
      ...process.env,
      NODE_ENV: 'test',
      DEBUG: 'pw:api',
    };

    return new Promise((resolve, reject) => {
      const testProcess = spawn('npx', playwrightArgs, {
        stdio: 'inherit',
        env,
        cwd: process.cwd()
      });

      let testOutput = '';

      testProcess.stdout?.on('data', (data) => {
        const output = data.toString();
        testOutput += output;
        this.parseTestOutput(output);
      });

      testProcess.stderr?.on('data', (data) => {
        const output = data.toString();
        console.error(`❌ Test Error: ${output}`);
      });

      testProcess.on('close', (code) => {
        this.endTime = Date.now();
        this.testResults.duration = this.endTime - this.startTime;

        this.log(`E2E tests completed with exit code: ${code}`, code === 0 ? 'success' : 'error');
        this.printSummary();

        if (code === 0) {
          resolve(this.testResults);
        } else {
          reject(new Error(`Tests failed with exit code ${code}`));
        }
      });

      testProcess.on('error', (error) => {
        this.log(`Failed to start test process: ${error.message}`, 'error');
        reject(error);
      });
    });
  }

  parseTestOutput(output) {
    // Parse Playwright test output for real-time feedback
    const lines = output.split('\n');

    for (const line of lines) {
      if (line.includes('✓')) {
        this.testResults.passed++;
        this.testResults.total++;
        this.log(`✅ ${line.replace('✓', '').trim()}`, 'success');
      } else if (line.includes('✗') || line.includes('failed')) {
        this.testResults.failed++;
        this.testResults.total++;
        this.log(`❌ ${line.trim()}`, 'error');
      } else if (line.includes('skipped')) {
        this.testResults.skipped++;
        this.testResults.total++;
        this.log(`⏭️  ${line.trim()}`, 'warning');
      } else if (line.includes('Running') && line.includes('test')) {
        this.log(`🚀 ${line.trim()}`, 'info');
      } else if (line.includes('Slow test')) {
        this.log(`🐌 ${line.trim()}`, 'warning');
      }
    }
  }

  printSummary() {
    const { total, passed, failed, skipped, duration } = this.testResults;

    this.log(`\n📊 Test Summary:`, 'info');
    this.log(`━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━`, 'info');
    this.log(`Total Tests: ${total}`, 'info');
    this.log(`✅ Passed: ${passed}`, 'success');
    this.log(`❌ Failed: ${failed}`, failed > 0 ? 'error' : 'info');
    this.log(`⏭️  Skipped: ${skipped}`, 'warning');
    this.log(`⏱️  Duration: ${(duration / 1000).toFixed(2)}s`, 'info');

    const passRate = total > 0 ? ((passed / total) * 100).toFixed(1) : 0;
    this.log(`📈 Pass Rate: ${passRate}%`, parseFloat(passRate) >= 80 ? 'success' : 'warning');

    // Performance analysis
    if (duration > 0) {
      const avgTestTime = (duration / total).toFixed(0);
      this.log(`⏱️  Avg Test Time: ${avgTestTime}ms per test`, 'info');
    }

    // Recommendations
    if (failed > 0) {
      this.log(`\n💡 Recommendations:`, 'warning');
      this.log(`   - Check test-results.json for detailed failure information`, 'info');
      this.log(`   - Run 'npm run test:e2e:ui' to debug failed tests interactively`, 'info');
      this.log(`   - Check browser console logs for client-side errors`, 'info');
    }

    if (passed > 0 && failed === 0) {
      this.log(`\n🎉 All tests passed! Your application is working correctly.`, 'success');
    }
  }

  async generateReport() {
    const reportPath = path.join(process.cwd(), 'test-results', 'e2e-report.json');

    try {
      const report = {
        timestamp: new Date().toISOString(),
        summary: this.testResults,
        environment: {
          nodeVersion: process.version,
          platform: process.platform,
          cwd: process.cwd(),
        },
        recommendations: this.generateRecommendations()
      };

      fs.writeFileSync(reportPath, JSON.stringify(report, null, 2));
      this.log(`📄 Detailed report saved to: ${reportPath}`, 'info');
    } catch (error) {
      this.log(`Failed to generate report: ${error.message}`, 'error');
    }
  }

  generateRecommendations() {
    const recommendations = [];

    if (this.testResults.failed > 0) {
      recommendations.push('Fix failing tests before deploying');
      recommendations.push('Check API endpoints are responding correctly');
      recommendations.push('Verify database connections are working');
    }

    if (this.testResults.duration > 300000) { // 5 minutes
      recommendations.push('Consider optimizing test performance - tests are running slow');
    }

    if (this.testResults.passed === 0) {
      recommendations.push('No tests are passing - check test setup and dependencies');
    }

    return recommendations;
  }
}

// CLI interface
async function main() {
  const args = process.argv.slice(2);
  const options = {};

  // Parse command line arguments
  args.forEach(arg => {
    if (arg.startsWith('--')) {
      const [key, value] = arg.slice(2).split('=');
      options[key] = value || true;
    }
  });

  const runner = new E2ETestRunner();

  try {
    await runner.checkPrerequisites();
    await runner.setupTestEnvironment();

    await runner.runTests(options);

    await runner.generateReport();

  } catch (error) {
    runner.log(`E2E test runner failed: ${error.message}`, 'error');
    process.exit(1);
  }
}

// Export for programmatic use
module.exports = E2ETestRunner;

// Run if called directly
if (require.main === module) {
  main().catch(error => {
    console.error('Fatal error:', error);
    process.exit(1);
  });
}
