#!/usr/bin/env python3
"""
Fetch REAL betting odds and player props from BallDontLie API.

This script fetches actual sportsbook odds (not estimated) for:
- NFL: Betting odds + Player props (2025 season, week 8+)
- NBA: Betting odds (when available)

Usage:
    python scripts/fetch_real_odds.py [sport]
    
    sport: nfl, nba, or all (default: all)
"""

import os
import sys
import json
import time
from datetime import datetime
from pathlib import Path

try:
    import requests
except ImportError:
    print("Error: requests library required. Install with: pip install requests")
    sys.exit(1)

# Configuration
API_BASE_URL = "https://api.balldontlie.io"
DATA_DIR = Path(__file__).parent.parent / "data" / "betting"
ODDS_DIR = Path(__file__).parent.parent / "data" / "odds"
PROPS_DIR = Path(__file__).parent.parent / "data" / "player_props"

API_KEY = os.environ.get("BALLDONTLIE_API_KEY", "db60bdff-30db-4e3c-999a-9ce31d0a6c6c")


def make_request(url: str) -> dict:
    """Make an API request to BallDontLie."""
    headers = {"Authorization": API_KEY}
    response = requests.get(url, headers=headers, timeout=30)
    response.raise_for_status()
    return response.json()


def fetch_nfl_odds(season: int = 2025, start_week: int = 8, end_week: int = 18) -> list:
    """Fetch NFL betting odds for specified weeks."""
    all_odds = []
    
    print(f"Fetching NFL odds for {season} season (weeks {start_week}-{end_week})...")
    
    for week in range(start_week, end_week + 1):
        url = f"{API_BASE_URL}/nfl/v1/odds?season={season}&week={week}&per_page=100"
        print(f"  Week {week}...", end=" ", flush=True)
        
        try:
            data = make_request(url)
            items = data.get("data", [])
            if items:
                # Add week info to each entry
                for item in items:
                    item["week"] = week
                    item["season"] = season
                all_odds.extend(items)
                print(f"{len(items)} odds")
            else:
                print("no data")
        except Exception as e:
            print(f"error: {e}")
        
        time.sleep(0.3)
    
    return all_odds


def fetch_nfl_player_props(game_ids: list) -> list:
    """Fetch NFL player props for specified games."""
    all_props = []
    
    print(f"\nFetching NFL player props for {len(game_ids)} games...")
    
    for i, game_id in enumerate(game_ids[:20]):  # Limit to 20 games
        url = f"{API_BASE_URL}/nfl/v1/odds/player_props?game_id={game_id}"
        print(f"  Game {game_id}...", end=" ", flush=True)
        
        try:
            data = make_request(url)
            items = data.get("data", [])
            if items:
                for item in items:
                    item["game_id"] = game_id
                all_props.extend(items)
                print(f"{len(items)} props")
            else:
                print("no props (game may be completed)")
        except Exception as e:
            print(f"error: {e}")
        
        time.sleep(0.3)
    
    return all_props


def fetch_nba_odds(dates: list = None) -> list:
    """Fetch NBA betting odds for specified dates."""
    all_odds = []
    
    if dates is None:
        # Default to recent dates
        from datetime import timedelta
        today = datetime.now()
        dates = [(today - timedelta(days=i)).strftime("%Y-%m-%d") for i in range(7)]
    
    print(f"Fetching NBA odds for {len(dates)} dates...")
    
    for date in dates:
        url = f"{API_BASE_URL}/v1/odds?date={date}"
        print(f"  {date}...", end=" ", flush=True)
        
        try:
            data = make_request(url)
            items = data.get("data", [])
            if items:
                for item in items:
                    item["date"] = date
                all_odds.extend(items)
                print(f"{len(items)} odds")
            else:
                print("no data")
        except Exception as e:
            print(f"error: {e}")
        
        time.sleep(0.3)
    
    return all_odds


def merge_odds_with_games(games_file: Path, odds: list, sport: str) -> list:
    """Merge real odds with game data."""
    if not games_file.exists():
        print(f"Games file not found: {games_file}")
        return []
    
    with open(games_file, "r") as f:
        games = json.load(f)
    
    # Create lookup by game_id
    odds_by_game = {}
    for odd in odds:
        game_id = odd.get("game_id")
        if game_id not in odds_by_game:
            odds_by_game[game_id] = []
        odds_by_game[game_id].append(odd)
    
    # Update games with real odds (use consensus/average)
    updated_count = 0
    for game in games:
        # Try to find matching game_id
        # Note: Our game IDs are hashed, so we need to match differently
        # For now, we'll save odds separately and link by date/teams
        pass
    
    return games


def save_odds_data(odds: list, sport: str):
    """Save odds data to file."""
    ODDS_DIR.mkdir(parents=True, exist_ok=True)
    
    output_file = ODDS_DIR / f"{sport}_odds.json"
    with open(output_file, "w") as f:
        json.dump(odds, f, indent=2)
    
    print(f"\nSaved {len(odds)} {sport.upper()} odds to {output_file}")
    
    # Also create a summary
    vendors = set(o.get("vendor") for o in odds if o.get("vendor"))
    games = set(o.get("game_id") for o in odds if o.get("game_id"))
    
    summary = {
        "sport": sport,
        "total_odds_entries": len(odds),
        "unique_games": len(games),
        "vendors": list(vendors),
        "imported_at": datetime.now().isoformat()
    }
    
    summary_file = ODDS_DIR / f"{sport}_odds_summary.json"
    with open(summary_file, "w") as f:
        json.dump(summary, f, indent=2)


def save_props_data(props: list, sport: str):
    """Save player props data to file."""
    PROPS_DIR.mkdir(parents=True, exist_ok=True)
    
    output_file = PROPS_DIR / f"{sport}_player_props.json"
    with open(output_file, "w") as f:
        json.dump(props, f, indent=2)
    
    print(f"Saved {len(props)} {sport.upper()} player props to {output_file}")


def process_nfl():
    """Process NFL odds and player props."""
    print("\n" + "=" * 60)
    print("Processing NFL Betting Data")
    print("=" * 60)
    
    # Fetch odds (2025 season, week 8+)
    odds = fetch_nfl_odds(season=2025, start_week=8, end_week=18)
    
    if odds:
        save_odds_data(odds, "nfl")
        
        # Get unique game IDs for player props
        game_ids = list(set(o.get("game_id") for o in odds if o.get("game_id")))
        
        # Fetch player props (live only, may be empty for completed games)
        props = fetch_nfl_player_props(game_ids)
        if props:
            save_props_data(props, "nfl")
    
    return {"odds": len(odds), "props": len(props) if 'props' in dir() else 0}


def process_nba():
    """Process NBA odds."""
    print("\n" + "=" * 60)
    print("Processing NBA Betting Data")
    print("=" * 60)
    
    # Fetch odds for recent dates
    from datetime import timedelta
    today = datetime.now()
    dates = [(today - timedelta(days=i)).strftime("%Y-%m-%d") for i in range(14)]
    
    odds = fetch_nba_odds(dates)
    
    if odds:
        save_odds_data(odds, "nba")
    
    return {"odds": len(odds)}


def main():
    """Main entry point."""
    print("=" * 60)
    print("BallDontLie REAL Odds & Player Props Fetcher")
    print("=" * 60)
    
    # Ensure directories exist
    ODDS_DIR.mkdir(parents=True, exist_ok=True)
    PROPS_DIR.mkdir(parents=True, exist_ok=True)
    
    # Determine which sports to process
    sports_to_process = ["nfl", "nba"]
    
    if len(sys.argv) > 1:
        arg = sys.argv[1].lower()
        if arg == "all":
            sports_to_process = ["nfl", "nba"]
        elif arg in ["nfl", "nba"]:
            sports_to_process = [arg]
        else:
            print(f"Unknown sport: {arg}")
            print("Available: nfl, nba, all")
            sys.exit(1)
    
    # Process each sport
    results = {}
    for sport in sports_to_process:
        if sport == "nfl":
            results["nfl"] = process_nfl()
        elif sport == "nba":
            results["nba"] = process_nba()
    
    # Summary
    print("\n" + "=" * 60)
    print("SUMMARY - Real Betting Data")
    print("=" * 60)
    for sport, data in results.items():
        print(f"  {sport.upper()}: {data.get('odds', 0)} odds entries", end="")
        if data.get('props'):
            print(f", {data['props']} player props")
        else:
            print()
    
    print("\n✅ Done! Real odds data saved.")
    print("\nNote: Player props are LIVE only - they may be empty for completed games.")


if __name__ == "__main__":
    main()
