#!/usr/bin/env python3
"""
Fetch ALL available real betting odds and player props from BallDontLie API.

Sports supported:
- NBA: Odds ✓, Player Props (check availability)
- NFL: Odds ✓, Player Props ✓ (LIVE only)
- NHL: Odds ✓, Player Props ✓ (LIVE only)
- MLB: Odds (seasonal), Player Props (check availability)

Usage:
    python scripts/fetch_all_odds.py
"""

import os
import sys
import json
import time
from datetime import datetime, timedelta
from pathlib import Path

try:
    import requests
except ImportError:
    print("Error: requests library required. Install with: pip install requests")
    sys.exit(1)

# Configuration
API_BASE_URL = "https://api.balldontlie.io"
DATA_DIR = Path(__file__).parent.parent / "data"
ODDS_DIR = DATA_DIR / "odds"
PROPS_DIR = DATA_DIR / "player_props"

API_KEY = os.environ.get("BALLDONTLIE_API_KEY", "db60bdff-30db-4e3c-999a-9ce31d0a6c6c")
HEADERS = {"Authorization": API_KEY}


def make_request(url: str, retries: int = 2) -> dict:
    """Make an API request with retries."""
    for attempt in range(retries + 1):
        try:
            response = requests.get(url, headers=HEADERS, timeout=30)
            if response.status_code == 429:  # Rate limited
                time.sleep(2)
                continue
            response.raise_for_status()
            return response.json()
        except Exception as e:
            if attempt == retries:
                raise
            time.sleep(1)
    return {}


def fetch_nba_odds(days: int = 60) -> list:
    """Fetch NBA betting odds for recent days."""
    all_odds = []
    today = datetime.now()
    
    print(f"\nFetching NBA odds (last {days} days)...")
    
    for i in range(days):
        date = (today - timedelta(days=i)).strftime("%Y-%m-%d")
        url = f"{API_BASE_URL}/v1/odds?date={date}&per_page=100"
        
        try:
            data = make_request(url)
            items = data.get("data", [])
            if items:
                for item in items:
                    item["date"] = date
                all_odds.extend(items)
                print(f"  {date}: {len(items)} odds")
            time.sleep(0.25)
        except Exception as e:
            print(f"  {date}: error - {e}")
    
    return all_odds


def fetch_nfl_odds(season: int = 2025) -> list:
    """Fetch NFL betting odds for season."""
    all_odds = []
    
    print(f"\nFetching NFL odds ({season} season)...")
    
    for week in range(1, 23):  # Regular + playoffs
        url = f"{API_BASE_URL}/nfl/v1/odds?season={season}&week={week}&per_page=100"
        
        try:
            data = make_request(url)
            items = data.get("data", [])
            if items:
                for item in items:
                    item["week"] = week
                    item["season"] = season
                all_odds.extend(items)
                print(f"  Week {week}: {len(items)} odds")
            time.sleep(0.25)
        except Exception as e:
            pass  # Week may not exist
    
    return all_odds


def fetch_nhl_odds(season: int = 2025) -> list:
    """Fetch NHL betting odds."""
    all_odds = []
    
    print(f"\nFetching NHL odds ({season} season)...")
    
    # Get recent dates
    today = datetime.now()
    for i in range(60):
        date = (today - timedelta(days=i)).strftime("%Y-%m-%d")
        url = f"{API_BASE_URL}/nhl/v1/odds?dates[]={date}&per_page=100"
        
        try:
            data = make_request(url)
            items = data.get("data", [])
            if items:
                for item in items:
                    item["date"] = date
                all_odds.extend(items)
                print(f"  {date}: {len(items)} odds")
            time.sleep(0.25)
        except Exception as e:
            pass
    
    return all_odds


def fetch_mlb_odds(season: int = 2024) -> list:
    """Fetch MLB betting odds."""
    all_odds = []
    
    print(f"\nFetching MLB odds ({season} season)...")
    
    # Try date-based approach
    # MLB season is April-October
    start_date = datetime(season, 4, 1)
    end_date = datetime(season, 10, 31)
    
    current = start_date
    while current <= end_date:
        date = current.strftime("%Y-%m-%d")
        url = f"{API_BASE_URL}/mlb/v1/odds?dates[]={date}&per_page=100"
        
        try:
            data = make_request(url)
            items = data.get("data", [])
            if items:
                for item in items:
                    item["date"] = date
                all_odds.extend(items)
                print(f"  {date}: {len(items)} odds")
            time.sleep(0.25)
        except Exception as e:
            pass
        
        current += timedelta(days=7)  # Check weekly
    
    return all_odds


def fetch_player_props(sport: str, game_ids: list) -> list:
    """Fetch player props for a sport."""
    all_props = []
    
    endpoints = {
        "nba": "/v1/odds/player_props",
        "nfl": "/nfl/v1/odds/player_props",
        "nhl": "/nhl/v1/odds/player_props",
        "mlb": "/mlb/v1/odds/player_props"
    }
    
    endpoint = endpoints.get(sport)
    if not endpoint:
        return []
    
    print(f"\nFetching {sport.upper()} player props ({len(game_ids)} games)...")
    
    for game_id in game_ids[:30]:  # Limit to 30 games
        url = f"{API_BASE_URL}{endpoint}?game_id={game_id}"
        
        try:
            data = make_request(url)
            items = data.get("data", [])
            if items:
                for item in items:
                    item["game_id"] = game_id
                all_props.extend(items)
                print(f"  Game {game_id}: {len(items)} props")
            else:
                print(f"  Game {game_id}: no props (may be completed)")
            time.sleep(0.3)
        except Exception as e:
            print(f"  Game {game_id}: {e}")
    
    return all_props


def get_upcoming_games(sport: str, limit: int = 30) -> list:
    """Get upcoming/recent games for player props."""
    endpoints = {
        "nba": "/v1/games",
        "nfl": "/nfl/v1/games",
        "nhl": "/nhl/v1/games",
        "mlb": "/mlb/v1/games"
    }
    
    endpoint = endpoints.get(sport)
    if not endpoint:
        return []
    
    # Try to get recent games
    today = datetime.now()
    games = []
    
    for i in range(-3, 7):  # 3 days ago to 7 days ahead
        date = (today + timedelta(days=i)).strftime("%Y-%m-%d")
        url = f"{API_BASE_URL}{endpoint}?dates[]={date}&per_page=50"
        
        try:
            data = make_request(url)
            items = data.get("data", [])
            games.extend(items)
            time.sleep(0.2)
        except:
            pass
    
    return games[:limit]


def save_data(data: list, filename: str, directory: Path):
    """Save data to JSON file."""
    directory.mkdir(parents=True, exist_ok=True)
    filepath = directory / filename
    
    with open(filepath, "w") as f:
        json.dump(data, f, indent=2)
    
    print(f"  Saved {len(data)} entries to {filepath}")


def create_summary(sport: str, odds: list, props: list):
    """Create summary for a sport."""
    vendors_odds = list(set(o.get("vendor") for o in odds if o.get("vendor")))
    vendors_props = list(set(p.get("vendor") for p in props if p.get("vendor")))
    games_odds = len(set(o.get("game_id") for o in odds if o.get("game_id")))
    
    summary = {
        "sport": sport,
        "odds": {
            "total_entries": len(odds),
            "unique_games": games_odds,
            "vendors": vendors_odds
        },
        "player_props": {
            "total_entries": len(props),
            "vendors": vendors_props
        },
        "imported_at": datetime.now().isoformat(),
        "data_source": "BallDontLie API (REAL SPORTSBOOK DATA)"
    }
    
    return summary


def main():
    """Main entry point."""
    print("=" * 70)
    print("BallDontLie - ALL Sports Real Odds & Player Props Fetcher")
    print("=" * 70)
    
    # Ensure directories exist
    ODDS_DIR.mkdir(parents=True, exist_ok=True)
    PROPS_DIR.mkdir(parents=True, exist_ok=True)
    
    results = {}
    
    # 1. NBA
    print("\n" + "=" * 50)
    print("NBA - National Basketball Association")
    print("=" * 50)
    
    nba_odds = fetch_nba_odds(days=60)
    nba_games = get_upcoming_games("nba")
    nba_game_ids = [g.get("id") for g in nba_games if g.get("id")]
    nba_props = fetch_player_props("nba", nba_game_ids)
    
    if nba_odds:
        save_data(nba_odds, "nba_odds.json", ODDS_DIR)
    if nba_props:
        save_data(nba_props, "nba_player_props.json", PROPS_DIR)
    
    results["nba"] = create_summary("nba", nba_odds, nba_props)
    
    # 2. NFL
    print("\n" + "=" * 50)
    print("NFL - National Football League")
    print("=" * 50)
    
    nfl_odds = fetch_nfl_odds(season=2025)
    nfl_games = get_upcoming_games("nfl")
    nfl_game_ids = [g.get("id") for g in nfl_games if g.get("id")]
    nfl_props = fetch_player_props("nfl", nfl_game_ids)
    
    if nfl_odds:
        save_data(nfl_odds, "nfl_odds.json", ODDS_DIR)
    if nfl_props:
        save_data(nfl_props, "nfl_player_props.json", PROPS_DIR)
    
    results["nfl"] = create_summary("nfl", nfl_odds, nfl_props)
    
    # 3. NHL
    print("\n" + "=" * 50)
    print("NHL - National Hockey League")
    print("=" * 50)
    
    nhl_odds = fetch_nhl_odds(season=2025)
    nhl_games = get_upcoming_games("nhl")
    nhl_game_ids = [g.get("id") for g in nhl_games if g.get("id")]
    nhl_props = fetch_player_props("nhl", nhl_game_ids)
    
    if nhl_odds:
        save_data(nhl_odds, "nhl_odds.json", ODDS_DIR)
    if nhl_props:
        save_data(nhl_props, "nhl_player_props.json", PROPS_DIR)
    
    results["nhl"] = create_summary("nhl", nhl_odds, nhl_props)
    
    # 4. MLB (check if available)
    print("\n" + "=" * 50)
    print("MLB - Major League Baseball")
    print("=" * 50)
    
    mlb_odds = fetch_mlb_odds(season=2024)
    mlb_games = get_upcoming_games("mlb")
    mlb_game_ids = [g.get("id") for g in mlb_games if g.get("id")]
    mlb_props = fetch_player_props("mlb", mlb_game_ids)
    
    if mlb_odds:
        save_data(mlb_odds, "mlb_odds.json", ODDS_DIR)
    if mlb_props:
        save_data(mlb_props, "mlb_player_props.json", PROPS_DIR)
    
    results["mlb"] = create_summary("mlb", mlb_odds, mlb_props)
    
    # Save master summary
    master_summary = {
        "fetched_at": datetime.now().isoformat(),
        "data_source": "BallDontLie API - REAL SPORTSBOOK ODDS",
        "sports": results,
        "totals": {
            "total_odds": sum(r["odds"]["total_entries"] for r in results.values()),
            "total_props": sum(r["player_props"]["total_entries"] for r in results.values())
        }
    }
    
    with open(ODDS_DIR / "all_odds_summary.json", "w") as f:
        json.dump(master_summary, f, indent=2)
    
    # Print final summary
    print("\n" + "=" * 70)
    print("FINAL SUMMARY - Real Betting Data from BallDontLie API")
    print("=" * 70)
    
    for sport, data in results.items():
        print(f"\n{sport.upper()}:")
        print(f"  Odds: {data['odds']['total_entries']:,} entries from {len(data['odds']['vendors'])} sportsbooks")
        print(f"  Player Props: {data['player_props']['total_entries']:,} entries")
        if data['odds']['vendors']:
            print(f"  Vendors: {', '.join(data['odds']['vendors'][:5])}...")
    
    print(f"\n{'=' * 70}")
    print(f"TOTALS: {master_summary['totals']['total_odds']:,} odds + {master_summary['totals']['total_props']:,} player props")
    print(f"{'=' * 70}")
    print("\n✅ All data saved to /data/odds/ and /data/player_props/")


if __name__ == "__main__":
    main()
