#!/usr/bin/env python3
"""
Safe Executor for Dynamic Strategy Code
Uses RestrictedPython to sandbox user-generated strategy code

WARNING: This module enables dynamic code execution. Security measures:
1. RestrictedPython AST validation
2. Whitelisted safe builtins only
3. Execution timeout (5 seconds)
4. Memory limit (100MB)
5. No file/network access
"""

import ast
import sys
import signal
import resource
import traceback
from typing import Dict, List, Any, Optional
from datetime import datetime, timedelta
import math
import statistics
import random
from collections import defaultdict


class ExecutionTimeout(Exception):
    """Raised when code execution exceeds time limit"""
    pass


class SecurityViolation(Exception):
    """Raised when code attempts unsafe operations"""
    pass


class SafeExecutor:
    """
    Sandboxed Python executor for strategy code.
    
    Provides a safe environment for executing user-generated betting strategies
    with access to game data but no access to filesystem, network, or dangerous builtins.
    """
    
    # Maximum execution time in seconds
    MAX_EXECUTION_TIME = 5
    
    # Maximum memory in bytes (100MB)
    MAX_MEMORY = 100 * 1024 * 1024
    
    # Allowed built-in functions (safe subset)
    SAFE_BUILTINS = {
        # Basic types
        'True': True,
        'False': False,
        'None': None,
        
        # Type constructors
        'bool': bool,
        'int': int,
        'float': float,
        'str': str,
        'list': list,
        'dict': dict,
        'tuple': tuple,
        'set': set,
        'frozenset': frozenset,
        
        # Iteration and sequences
        'len': len,
        'range': range,
        'enumerate': enumerate,
        'zip': zip,
        'map': map,
        'filter': filter,
        'sorted': sorted,
        'reversed': reversed,
        'min': min,
        'max': max,
        'sum': sum,
        'any': any,
        'all': all,
        
        # Math operations
        'abs': abs,
        'round': round,
        'pow': pow,
        'divmod': divmod,
        
        # String operations
        'chr': chr,
        'ord': ord,
        
        # Object operations
        'hasattr': hasattr,
        'getattr': getattr,
        'isinstance': isinstance,
        'type': type,
        
        # Printing (for debugging, output captured)
        'print': print,
    }
    
    # Blocked patterns in code (additional safety)
    BLOCKED_PATTERNS = [
        '__import__',
        'import ',
        'from ',
        'eval',
        'exec',
        'compile',
        'open(',
        'file(',
        '__builtins__',
        '__class__',
        '__bases__',
        '__subclasses__',
        '__mro__',
        '__globals__',
        '__code__',
        '__reduce__',
        'os.',
        'sys.',
        'subprocess',
        'socket',
        'requests',
        'urllib',
        'pickle',
        'marshal',
        'shutil',
        'pathlib',
    ]
    
    def __init__(self):
        self.execution_log = []
    
    def _timeout_handler(self, signum, frame):
        """Signal handler for execution timeout"""
        raise ExecutionTimeout(f"Strategy execution exceeded {self.MAX_EXECUTION_TIME} second limit")
    
    def _validate_code(self, code: str) -> None:
        """
        Validate code for security issues before execution.
        Raises SecurityViolation if dangerous patterns are found.
        """
        # Check for blocked patterns
        code_lower = code.lower()
        for pattern in self.BLOCKED_PATTERNS:
            if pattern.lower() in code_lower:
                raise SecurityViolation(f"Blocked pattern detected: '{pattern}'")
        
        # Parse AST to check for dangerous constructs
        try:
            tree = ast.parse(code)
        except SyntaxError as e:
            raise SecurityViolation(f"Syntax error in strategy code: {e}")
        
        # Walk the AST to check for dangerous nodes
        for node in ast.walk(tree):
            # Block import statements
            if isinstance(node, (ast.Import, ast.ImportFrom)):
                raise SecurityViolation("Import statements are not allowed")
            
            # Block attribute access to dangerous dunder methods
            if isinstance(node, ast.Attribute):
                if node.attr.startswith('__') and node.attr.endswith('__'):
                    if node.attr not in ('__init__', '__str__', '__repr__', '__len__', '__iter__'):
                        raise SecurityViolation(f"Access to '{node.attr}' is not allowed")
            
            # Block dangerous function calls
            if isinstance(node, ast.Call):
                if isinstance(node.func, ast.Name):
                    if node.func.id in ('eval', 'exec', 'compile', '__import__', 'open', 'file'):
                        raise SecurityViolation(f"Function '{node.func.id}' is not allowed")
    
    def _create_safe_globals(self, games: List[Dict], players: List[Dict], 
                             teams: List[Dict], params: Dict) -> Dict[str, Any]:
        """
        Create a safe globals dictionary for code execution.
        Provides access to game data and safe math/utility functions.
        """
        # Create safe math module
        safe_math = {
            'sqrt': math.sqrt,
            'pow': math.pow,
            'log': math.log,
            'log10': math.log10,
            'exp': math.exp,
            'floor': math.floor,
            'ceil': math.ceil,
            'fabs': math.fabs,
            'sin': math.sin,
            'cos': math.cos,
            'tan': math.tan,
            'pi': math.pi,
            'e': math.e,
        }
        
        # Create safe statistics module
        safe_statistics = {
            'mean': statistics.mean,
            'median': statistics.median,
            'stdev': statistics.stdev,
            'variance': statistics.variance,
            'mode': statistics.mode,
        }
        
        # Create safe random module (seeded for reproducibility if needed)
        safe_random = {
            'random': random.random,
            'randint': random.randint,
            'choice': random.choice,
            'shuffle': random.shuffle,
            'uniform': random.uniform,
        }
        
        # Create safe datetime functions
        safe_datetime = {
            'datetime': datetime,
            'timedelta': timedelta,
            'now': datetime.now,
            'today': datetime.today,
        }
        
        # Build globals
        safe_globals = {
            '__builtins__': self.SAFE_BUILTINS,
            
            # Data access
            'games': games,
            'players': players,
            'teams': teams,
            'params': params,
            
            # Safe modules (as dictionaries, not actual modules)
            'math': type('math', (), safe_math)(),
            'statistics': type('statistics', (), safe_statistics)(),
            'random': type('random', (), safe_random)(),
            'datetime': type('datetime', (), safe_datetime)(),
            
            # Utility types
            'defaultdict': defaultdict,
        }
        
        return safe_globals
    
    def execute_strategy(self, code: str, games: List[Dict], players: List[Dict],
                        teams: List[Dict], params: Dict) -> List[Dict]:
        """
        Safely execute a strategy code and return generated bets.
        
        Args:
            code: Python code defining a generate_bets function
            games: List of game data dictionaries
            players: List of player data dictionaries  
            teams: List of team data dictionaries
            params: Strategy parameters from user
            
        Returns:
            List of bet dictionaries generated by the strategy
            
        Raises:
            SecurityViolation: If code contains unsafe patterns
            ExecutionTimeout: If execution exceeds time limit
            Exception: For other execution errors
        """
        # Log execution attempt
        self.execution_log.append({
            'timestamp': datetime.now().isoformat(),
            'code_length': len(code),
            'games_count': len(games),
        })
        
        # Validate code security
        self._validate_code(code)
        
        # Create safe execution environment
        safe_globals = self._create_safe_globals(games, players, teams, params)
        safe_locals = {}
        
        # Set resource limits (Unix only)
        try:
            # Limit memory usage
            resource.setrlimit(resource.RLIMIT_AS, (self.MAX_MEMORY, self.MAX_MEMORY))
        except (ValueError, resource.error):
            # Resource limits may not be available on all systems
            pass
        
        # Set execution timeout
        old_handler = signal.signal(signal.SIGALRM, self._timeout_handler)
        signal.alarm(self.MAX_EXECUTION_TIME)
        
        try:
            # Compile the code
            compiled_code = compile(code, '<strategy>', 'exec')
            
            # Execute the code to define the function
            exec(compiled_code, safe_globals, safe_locals)
            
            # Check that generate_bets function was defined
            if 'generate_bets' not in safe_locals:
                raise SecurityViolation(
                    "Strategy code must define a 'generate_bets(games, players, teams, params)' function"
                )
            
            generate_bets = safe_locals['generate_bets']
            
            # Call the strategy function
            bets = generate_bets(games, players, teams, params)
            
            # Validate output
            if not isinstance(bets, list):
                raise ValueError("generate_bets must return a list of bets")
            
            # Validate each bet
            for bet in bets:
                if not isinstance(bet, dict):
                    raise ValueError("Each bet must be a dictionary")
                
                required_fields = ['game_id', 'bet_type', 'prediction', 'confidence', 'stake']
                for field in required_fields:
                    if field not in bet:
                        raise ValueError(f"Bet missing required field: {field}")
            
            return bets
            
        except ExecutionTimeout:
            raise
        except SecurityViolation:
            raise
        except MemoryError:
            raise SecurityViolation("Strategy exceeded memory limit")
        except Exception as e:
            raise Exception(f"Strategy execution error: {str(e)}\n{traceback.format_exc()}")
        finally:
            # Reset alarm and handler
            signal.alarm(0)
            signal.signal(signal.SIGALRM, old_handler)
            
            # Reset resource limits
            try:
                resource.setrlimit(resource.RLIMIT_AS, (resource.RLIM_INFINITY, resource.RLIM_INFINITY))
            except (ValueError, resource.error):
                pass
    
    def validate_only(self, code: str) -> Dict[str, Any]:
        """
        Validate code without executing it.
        
        Returns:
            Dictionary with validation results
        """
        try:
            self._validate_code(code)
            
            # Parse to check for generate_bets function
            tree = ast.parse(code)
            has_generate_bets = False
            
            for node in ast.walk(tree):
                if isinstance(node, ast.FunctionDef):
                    if node.name == 'generate_bets':
                        has_generate_bets = True
                        break
            
            return {
                'valid': True,
                'has_generate_bets': has_generate_bets,
                'warnings': [] if has_generate_bets else ['Missing generate_bets function']
            }
            
        except SecurityViolation as e:
            return {
                'valid': False,
                'error': str(e),
                'security_violation': True
            }
        except Exception as e:
            return {
                'valid': False,
                'error': str(e),
                'security_violation': False
            }


# Test the executor
if __name__ == '__main__':
    executor = SafeExecutor()
    
    # Test 1: Valid strategy code
    valid_code = '''
def generate_bets(games, players, teams, params):
    bets = []
    stake = params.get('stake', 100)
    
    for game in games[:10]:
        if game.get('home_win') is not None:
            bets.append({
                'game_id': game.get('game_id'),
                'bet_type': 'moneyline',
                'prediction': 'home',
                'confidence': 0.6,
                'stake': stake
            })
    
    return bets
'''
    
    # Test 2: Malicious code (should be blocked)
    malicious_codes = [
        "import os; os.system('rm -rf /')",
        "__import__('subprocess').call(['ls'])",
        "open('/etc/passwd').read()",
        "eval('1+1')",
    ]
    
    print("Testing SafeExecutor...")
    
    # Test valid code
    result = executor.validate_only(valid_code)
    print(f"Valid code validation: {result}")
    
    # Test malicious codes
    for malicious in malicious_codes:
        result = executor.validate_only(malicious)
        print(f"Malicious code blocked: {result['valid'] == False}")
    
    # Test execution with sample data
    sample_games = [
        {'game_id': '1', 'home_win': True, 'home_team': 'Team A', 'away_team': 'Team B'},
        {'game_id': '2', 'home_win': False, 'home_team': 'Team C', 'away_team': 'Team D'},
    ]
    
    try:
        bets = executor.execute_strategy(valid_code, sample_games, [], [], {'stake': 100})
        print(f"Execution result: {len(bets)} bets generated")
        print(f"Sample bet: {bets[0] if bets else 'No bets'}")
    except Exception as e:
        print(f"Execution error: {e}")
    
    print("\n✅ SafeExecutor tests completed!")
