/**
 * Database Setup Script for Backtesting AI Platform
 * Run this script to initialize your MySQL database with the proper schema
 *
 * Usage:
 * 1. Update your .env file with the correct DATABASE_URL
 * 2. Run: node prisma/setup-db.js
 */

const { PrismaClient } = require('@prisma/client');
const { execSync } = require('child_process');

async function setupDatabase() {
  console.log('🚀 Setting up Backtesting AI Platform Database...\n');

  // Check if .env file exists
  const fs = require('fs');
  const path = require('path');

  const envPath = path.join(__dirname, '..', '.env');
  if (!fs.existsSync(envPath)) {
    console.log('❌ .env file not found!');
    console.log('📝 Please create a .env file with your database configuration:');
    console.log('');
    console.log('DATABASE_URL="mysql://username:password@localhost:3306/backtest_db"');
    console.log('');
    console.log('Replace with your actual MySQL credentials.');
    process.exit(1);
  }

  const prisma = new PrismaClient();

  try {
    // Test database connection
    console.log('🔍 Testing database connection...');
    await prisma.$connect();
    console.log('✅ Database connection successful!\n');

    // Run database migrations
    console.log('📦 Running database migrations...');
    execSync('npx prisma db push', { stdio: 'inherit' });
    console.log('✅ Database schema created!\n');

    // Seed with initial data
    console.log('🌱 Seeding initial data...');
    await seedInitialData(prisma);
    console.log('✅ Initial data seeded!\n');

    console.log('🎉 Database setup complete!');
    console.log('');
    console.log('Next steps:');
    console.log('1. Start your Next.js app: npm run dev');
    console.log('2. Start the RAG service: npm run rag:start');
    console.log('3. Or run both: npm run dev:full');

  } catch (error) {
    console.error('❌ Database setup failed:', error.message);
    process.exit(1);
  } finally {
    await prisma.$disconnect();
  }
}

async function seedInitialData(prisma) {
  // Create some sample strategies as templates
  const sampleStrategies = [
    {
      name: 'Basic Momentum Strategy',
      description: 'A simple momentum-based trading strategy for stocks',
      userId: 'system', // System-generated templates
      domain: 'STOCKS',
      rules: {
        domain: 'stocks',
        lookback: 20,
        trigger: 'momentumDip',
        threshold: 0.05,
        direction: 'long',
        description: 'Buy when momentum dips below threshold'
      },
      isPublic: true,
      isTemplate: true,
      tags: ['momentum', 'trend-following', 'beginner']
    },
    {
      name: 'Sports Parlay Template',
      description: 'Template for creating multi-leg sports betting parlays',
      userId: 'system',
      domain: 'SPORTS',
      rules: {
        domain: 'sports',
        trigger: 'parlay',
        stakePerLeg: 10,
        description: 'Multi-leg parlay with $10 stake per leg'
      },
      isPublic: true,
      isTemplate: true,
      tags: ['parlay', 'multi-leg', 'sports-betting']
    },
    {
      name: 'Crypto Mean Reversion',
      description: 'Mean reversion strategy for cryptocurrency markets',
      userId: 'system',
      domain: 'CRYPTO',
      rules: {
        domain: 'crypto',
        lookback: 14,
        trigger: 'bounceAfterLosses',
        threshold: 3,
        direction: 'long',
        description: 'Buy after 3 consecutive losses (mean reversion)'
      },
      isPublic: true,
      isTemplate: true,
      tags: ['mean-reversion', 'crypto', 'contrarian']
    }
  ];

  for (const strategy of sampleStrategies) {
    await prisma.strategy.create({ data: strategy });
  }

  console.log(`Created ${sampleStrategies.length} sample strategy templates`);
}

if (require.main === module) {
  setupDatabase().catch(console.error);
}

module.exports = { setupDatabase };
