#!/usr/bin/env python3
"""
NBA Backtesting Engine for Sports Betting Strategies
Runs actual backtests on NBA data with comprehensive results
"""

import os
import sys
import json
import pandas as pd
import numpy as np
from datetime import datetime, timedelta
from typing import Dict, List, Any, Optional, Callable
import random

# Add the virtual environment to Python path
sys.path.insert(0, os.path.join(os.path.dirname(__file__), 'nba_api_env', 'lib', 'python3.13', 'site-packages'))

class NBABacktestingEngine:
    """NBA-specific backtesting engine for sports betting strategies"""

    def __init__(self):
        self.nba_data_service = None
        self.available_strategies = {
            'player_points_over': self._strategy_player_points_over,
            'team_win_streak': self._strategy_team_win_streak,
            'home_away_performance': self._strategy_home_away_performance,
            'player_vs_defense': self._strategy_player_vs_defense,
            'injury_impact': self._strategy_injury_impact,
            'momentum_reversal': self._strategy_momentum_reversal,
            'spread_betting': self._strategy_spread_betting,
            'moneyline_prediction': self._strategy_moneyline_prediction,
        }

    def run_backtest(self, strategy_name: str, parameters: Dict[str, Any],
                    season: str = "2023-24", min_games: int = 10) -> Dict[str, Any]:
        """Run a backtest on NBA data with given strategy and parameters"""

        if strategy_name not in self.available_strategies:
            return {
                'error': f'Unknown strategy: {strategy_name}',
                'available_strategies': list(self.available_strategies.keys())
            }

        try:
            # Get NBA data for the season
            games_data = self._get_games_data(season)
            players_data = self._get_players_data()
            teams_data = self._get_teams_data()

            if not games_data:
                return {'error': 'No games data available for backtesting'}

            # Run the strategy
            strategy_func = self.available_strategies[strategy_name]
            trades = strategy_func(games_data, players_data, teams_data, parameters)

            if len(trades) < min_games:
                return {
                    'error': f'Insufficient data: Only {len(trades)} trades generated, minimum {min_games} required',
                    'trades_generated': len(trades)
                }

            # Calculate comprehensive results
            results = self._calculate_backtest_results(trades, parameters)

            return {
                'success': True,
                'strategy_name': strategy_name,
                'parameters': parameters,
                'season': season,
                'total_trades': len(trades),
                'results': results,
                'trades': trades[-50:],  # Last 50 trades for detailed analysis
                'performance_summary': self._generate_performance_summary(results),
                'risk_metrics': self._calculate_risk_metrics(trades)
            }

        except Exception as e:
            return {
                'error': f'Backtest failed: {str(e)}',
                'strategy_name': strategy_name,
                'parameters': parameters
            }

    def _get_games_data(self, season: str) -> List[Dict[str, Any]]:
        """Get NBA games data for backtesting"""
        try:
            # Call the NBA data service
            import subprocess
            result = subprocess.run([
                sys.executable, 'nba_data_service.py', 'games',
                json.dumps({'use_cache': True, 'season': season, 'limit': 1000})
            ], capture_output=True, text=True, cwd=os.path.dirname(__file__))

            if result.returncode == 0:
                return json.loads(result.stdout.strip())
            else:
                print(f"Error getting games data: {result.stderr}")
                return []
        except Exception as e:
            print(f"Failed to get games data: {e}")
            return []

    def _get_players_data(self) -> List[Dict[str, Any]]:
        """Get NBA players data"""
        try:
            import subprocess
            result = subprocess.run([
                sys.executable, 'nba_data_service.py', 'players',
                json.dumps({'use_cache': True, 'limit': 100})
            ], capture_output=True, text=True, cwd=os.path.dirname(__file__))

            if result.returncode == 0:
                return json.loads(result.stdout.strip())
            else:
                return []
        except Exception as e:
            return []

    def _get_teams_data(self) -> List[Dict[str, Any]]:
        """Get NBA teams data"""
        try:
            import subprocess
            result = subprocess.run([
                sys.executable, 'nba_data_service.py', 'teams',
                json.dumps({'use_cache': True})
            ], capture_output=True, text=True, cwd=os.path.dirname(__file__))

            if result.returncode == 0:
                return json.loads(result.stdout.strip())
            else:
                return []
        except Exception as e:
            return []

    # NBA-Specific Strategies

    def _strategy_player_points_over(self, games: List[Dict], players: List[Dict],
                                   teams: List[Dict], params: Dict) -> List[Dict]:
        """Strategy: Bet on player points over a threshold.
        
        NOTE: This strategy currently uses SIMULATED player points data.
        For real backtesting, integrate with:
        - NBA API PlayerGameLogs endpoint to get actual points per game
        - Or use SportsData.io player statistics API
        
        TODO: Replace np.random.normal() with actual player game logs.
        """
        trades = []
        player_name = params.get('player_name', '').lower()
        points_threshold = params.get('points_threshold', 25)
        stake = float(params.get('stake', 100))

        # Find the player
        target_player = None
        for player in players:
            if player_name in player.get('full_name', '').lower():
                target_player = player
                break

        if not target_player:
            return trades

        # TODO: Replace with actual player game logs from NBA API
        # Example: playergamelogs.PlayerGameLogs(player_id=target_player['id'])
        # For now, using simulated data as placeholder
        for game in games[-200:]:  # Last 200 games for backtesting
            # SIMULATED player performance - replace with real game logs
            simulated_points = np.random.normal(25, 8)  # Mean 25, std dev 8

            trade = {
                'id': f"player_points_{len(trades)}",
                'date': game.get('game_date', '2024-01-01'),
                'strategy': 'player_points_over',
                'player': target_player.get('full_name'),
                'threshold': points_threshold,
                'actual_points': round(simulated_points, 1),
                'outcome': 'win' if simulated_points > points_threshold else 'loss',
                'profit': stake if simulated_points > points_threshold else -stake,
                'stake': stake,
                'game_info': f"{game.get('home_team_name', 'Unknown')} game",
                'confidence': 0.6,
                'data_source': 'SIMULATED - requires player game logs integration'
            }
            trades.append(trade)

        return trades

    def _strategy_team_win_streak(self, games: List[Dict], players: List[Dict],
                                teams: List[Dict], params: Dict) -> List[Dict]:
        """Strategy: Bet on teams with win streaks - uses real game outcomes"""
        trades = []
        min_streak = int(params.get('min_streak', 3))
        stake = float(params.get('stake', 100))

        # Track team win streaks
        team_streaks = {}
        processed_games = sorted(games, key=lambda x: x.get('game_date', ''))

        for game in processed_games[-300:]:  # Last 300 games
            home_team = game.get('home_team_name', '')
            away_team = game.get('away_team_name', '')
            
            # Skip games without complete data
            if not home_team or not away_team:
                continue
            if game.get('home_win') is None:
                continue

            # Use actual game result
            home_win = game.get('home_win', False)

            # Update streaks
            for team in [home_team, away_team]:
                if team not in team_streaks:
                    team_streaks[team] = 0

                if (team == home_team and home_win) or (team == away_team and not home_win):
                    team_streaks[team] += 1
                else:
                    team_streaks[team] = 0

            # Place bet on team with active streak
            if team_streaks.get(home_team, 0) >= min_streak:
                # Bet on home team to win
                outcome = 'win' if home_win else 'loss'
                trades.append({
                    'id': f"win_streak_{len(trades)}",
                    'date': game.get('game_date', '2024-01-01'),
                    'strategy': 'team_win_streak',
                    'team': home_team,
                    'streak': team_streaks[home_team],
                    'outcome': outcome,
                    'profit': stake if outcome == 'win' else -stake,
                    'stake': stake,
                    'game_info': f"{home_team} vs {away_team}",
                    'actual_score': f"{game.get('home_team_score')}-{game.get('away_team_score')}",
                    'confidence': min(0.8, 0.5 + (team_streaks[home_team] * 0.1))
                })

        return trades



    def _strategy_home_away_performance(self, games: List[Dict], players: List[Dict],
                                      teams: List[Dict], params: Dict) -> List[Dict]:
        """Strategy: Bet based on home/away performance - uses real game outcomes"""
        trades = []
        stake = float(params.get('stake', 100))
        focus_home = params.get('focus_home', True)  # True for home advantage, False for away

        # Track team home/away performance
        team_performance = {}

        for game in games[-400:]:  # Last 400 games
            home_team = game.get('home_team_name', '')
            away_team = game.get('away_team_name', '')
            
            # Skip games without complete data
            if not home_team or not away_team:
                continue
            if game.get('home_win') is None:
                continue

            # Initialize performance tracking
            for team in [home_team, away_team]:
                if team not in team_performance:
                    team_performance[team] = {'home': {'wins': 0, 'total': 0}, 'away': {'wins': 0, 'total': 0}}

            # Use actual game outcome
            home_win = game.get('home_win', False)

            # Update performance
            team_performance[home_team]['home']['total'] += 1
            team_performance[away_team]['away']['total'] += 1

            if home_win:
                team_performance[home_team]['home']['wins'] += 1
            else:
                team_performance[away_team]['away']['wins'] += 1

            # Place bet based on historical performance
            if focus_home and team_performance[home_team]['home']['total'] >= 5:
                home_win_rate = team_performance[home_team]['home']['wins'] / team_performance[home_team]['home']['total']
                if home_win_rate > 0.6:  # Home team has >60% win rate at home
                    outcome = 'win' if home_win else 'loss'
                    trades.append({
                        'id': f"home_perf_{len(trades)}",
                        'date': game.get('game_date', '2024-01-01'),
                        'strategy': 'home_away_performance',
                        'team': home_team,
                        'venue': 'home',
                        'win_rate': round(home_win_rate, 3),
                        'outcome': outcome,
                        'profit': stake if outcome == 'win' else -stake,
                        'stake': stake,
                        'game_info': f"{home_team} at home vs {away_team}",
                        'actual_score': f"{game.get('home_team_score')}-{game.get('away_team_score')}",
                        'confidence': home_win_rate
                    })

        return trades


    def _strategy_player_vs_defense(self, games: List[Dict], players: List[Dict],
                                  teams: List[Dict], params: Dict) -> List[Dict]:
        """Strategy: Player performance vs opponent defense"""
        trades = []
        player_name = params.get('player_name', '').lower()
        stake = float(params.get('stake', 100))

        # Find the player
        target_player = None
        for player in players:
            if player_name in player.get('full_name', '').lower():
                target_player = player
                break

        if not target_player:
            return trades

        # Simulate defense ratings for teams
        team_defense = {team.get('abbreviation', ''): random.uniform(0.8, 1.2) for team in teams}

        for game in games[-150:]:  # Last 150 games
            home_team = game.get('home_team_name', '')
            away_team = game.get('away_team_id', '')

            # Determine opponent and venue
            opponent = away_team if random.choice([True, False]) else home_team
            venue = 'home' if opponent == away_team else 'away'

            defense_rating = team_defense.get(opponent, 1.0)

            # Simulate performance (better against weaker defense)
            base_performance = 25
            performance_multiplier = 1 / defense_rating  # Better against weak defense
            simulated_points = np.random.normal(base_performance * performance_multiplier, 6)

            threshold = 22  # Points over/under line
            outcome = 'win' if simulated_points > threshold else 'loss'

            trades.append({
                'id': f"vs_defense_{len(trades)}",
                'date': game.get('game_date', '2024-01-01'),
                'strategy': 'player_vs_defense',
                'player': target_player.get('full_name'),
                'opponent': opponent,
                'venue': venue,
                'defense_rating': round(defense_rating, 2),
                'actual_points': round(simulated_points, 1),
                'threshold': threshold,
                'outcome': outcome,
                'profit': stake if outcome == 'win' else -stake,
                'stake': stake,
                'game_info': f"{target_player.get('full_name')} vs {opponent}",
                'confidence': max(0.5, 1 - defense_rating)
            })

        return trades

    def _strategy_injury_impact(self, games: List[Dict], players: List[Dict],
                              teams: List[Dict], params: Dict) -> List[Dict]:
        """Strategy: Bet on injury impact on team performance - uses real game outcomes.
        
        Note: This strategy still simulates injury impact since real injury data 
        isn't available in the basic game data. However, the game outcomes are real.
        For production use, integrate with an injury data API.
        """
        trades = []
        team_name = params.get('team_name', '').lower()
        stake = float(params.get('stake', 100))
        bet_home = params.get('bet_home', True)  # Bet on home team performance

        # Find the team
        target_team = None
        for team in teams:
            if team_name in team.get('full_name', '').lower():
                target_team = team
                break

        if not target_team:
            return trades

        for game in games[-200:]:  # Last 200 games
            home_team = game.get('home_team_name', '')
            away_team = game.get('away_team_name', '')
            
            # Skip games without complete data
            if game.get('home_win') is None:
                continue

            # Only bet on games involving our target team
            team_full_name = target_team.get('full_name', '')
            is_home = team_full_name in home_team
            is_away = team_full_name in away_team
            
            if not is_home and not is_away:
                continue

            # Use actual game result
            home_win = game.get('home_win', False)
            
            # Did our target team win?
            team_won = (is_home and home_win) or (is_away and not home_win)
            
            # If betting on team to win, outcome depends on actual result
            outcome = 'win' if team_won else 'loss'

            trades.append({
                'id': f"injury_impact_{len(trades)}",
                'date': game.get('game_date', '2024-01-01'),
                'strategy': 'injury_impact',
                'team': team_full_name,
                'venue': 'home' if is_home else 'away',
                'outcome': outcome,
                'profit': stake if outcome == 'win' else -stake,
                'stake': stake,
                'game_info': f"{team_full_name} ({'home' if is_home else 'away'}) vs {away_team if is_home else home_team}",
                'actual_score': f"{game.get('home_team_score')}-{game.get('away_team_score')}",
                'confidence': 0.55
            })

        return trades

    def _strategy_momentum_reversal(self, games: List[Dict], players: List[Dict],
                                  teams: List[Dict], params: Dict) -> List[Dict]:
        """Strategy: Bet on momentum reversal after streaks - uses real game outcomes"""
        trades = []
        streak_length = int(params.get('streak_length', 3))
        stake = float(params.get('stake', 100))

        # Track team recent performance
        recent_games = {}
        # Store actual game results to look ahead
        game_results = {}

        # First pass: build game result lookup
        sorted_games = sorted(games, key=lambda x: x.get('game_date', ''))
        for i, game in enumerate(sorted_games):
            home_team = game.get('home_team_name', '')
            away_team = game.get('away_team_name', '')
            if game.get('home_win') is None:
                continue
            game_results[i] = {
                'home_team': home_team,
                'away_team': away_team,
                'home_win': game.get('home_win', False)
            }

        # Second pass: evaluate momentum strategy
        for i, game in enumerate(sorted_games[-500:]):
            home_team = game.get('home_team_name', '')
            away_team = game.get('away_team_name', '')
            
            # Skip games without complete data
            if not home_team or not away_team:
                continue
            if game.get('home_win') is None:
                continue

            # Use actual game result
            home_win = game.get('home_win', False)

            # Initialize tracking
            for team in [home_team, away_team]:
                if team not in recent_games:
                    recent_games[team] = []

            # Update recent games with actual results
            recent_games[home_team].append(home_win)
            recent_games[away_team].append(not home_win)

            # Keep only last 10 games
            for team in [home_team, away_team]:
                if len(recent_games[team]) > 10:
                    recent_games[team] = recent_games[team][-10:]

            # Check for betting opportunities (bet BEFORE this game, evaluate with actual result)
            for team in [home_team, away_team]:
                if len(recent_games[team]) > streak_length:  # Need history before this game
                    # Results before this game
                    pre_game_results = recent_games[team][-(streak_length+1):-1]
                    is_streak = all(pre_game_results) or not any(pre_game_results)

                    if is_streak and len(pre_game_results) == streak_length:
                        is_winning_streak = pre_game_results[0]
                        bet_on_win = not is_winning_streak  # Reverse the momentum

                        # Actual result for this team in current game
                        actual_win = (team == home_team and home_win) or (team == away_team and not home_win)
                        outcome = 'win' if (actual_win == bet_on_win) else 'loss'

                        trades.append({
                            'id': f"momentum_{len(trades)}",
                            'date': game.get('game_date', '2024-01-01'),
                            'strategy': 'momentum_reversal',
                            'team': team,
                            'streak_type': 'winning' if is_winning_streak else 'losing',
                            'streak_length': streak_length,
                            'outcome': outcome,
                            'profit': stake if outcome == 'win' else -stake,
                            'stake': stake,
                            'game_info': f"{team} momentum reversal after {streak_length} game streak",
                            'actual_score': f"{game.get('home_team_score')}-{game.get('away_team_score')}",
                            'confidence': 0.55
                        })

        return trades

    def _strategy_spread_betting(self, games: List[Dict], players: List[Dict],
                               teams: List[Dict], params: Dict) -> List[Dict]:
        """Strategy: Point spread betting - uses real game scores"""
        trades = []
        stake = float(params.get('stake', 100))
        spread_line = params.get('spread_line', 0)  # The spread to bet against (e.g., -5.5 means home favored by 5.5)

        for game in games[-300:]:  # Last 300 games
            home_score = game.get('home_team_score')
            away_score = game.get('away_team_score')
            
            # Skip games without complete score data
            if home_score is None or away_score is None:
                continue

            # Use real point margin from the game
            actual_margin = home_score - away_score
            
            # Bet on home team to cover the spread
            # If spread_line is -5.5, home team must win by more than 5.5 to cover
            covers_spread = actual_margin > spread_line

            outcome = 'win' if covers_spread else 'loss'

            trades.append({
                'id': f"spread_{len(trades)}",
                'date': game.get('game_date', '2024-01-01'),
                'strategy': 'spread_betting',
                'home_team': game.get('home_team_name', ''),
                'away_team': game.get('away_team_name', ''),
                'spread': spread_line,
                'home_score': home_score,
                'away_score': away_score,
                'margin': actual_margin,
                'outcome': outcome,
                'profit': stake if outcome == 'win' else -stake,
                'stake': stake,
                'game_info': f"{game.get('home_team_name', '')} {spread_line:+.1f} vs {game.get('away_team_name', '')}",
                'confidence': 0.52
            })

        return trades

    def _strategy_moneyline_prediction(self, games: List[Dict], players: List[Dict],
                                     teams: List[Dict], params: Dict) -> List[Dict]:
        """Strategy: Moneyline winner prediction - uses real game outcomes"""
        trades = []
        stake = float(params.get('stake', 100))
        bet_home = params.get('bet_home', True)  # True to bet home team, False to bet away

        for game in games[-400:]:  # Last 400 games
            # Skip games without complete data
            if game.get('home_win') is None:
                continue
            if not game.get('home_team_name') or not game.get('away_team_name'):
                continue

            # Use actual game result
            home_win = game.get('home_win', False)
            
            # Determine outcome based on bet direction
            if bet_home:
                outcome = 'win' if home_win else 'loss'
                bet_team = game.get('home_team_name', '')
            else:
                outcome = 'win' if not home_win else 'loss'
                bet_team = game.get('away_team_name', '')

            trades.append({
                'id': f"moneyline_{len(trades)}",
                'date': game.get('game_date', '2024-01-01'),
                'strategy': 'moneyline_prediction',
                'team': bet_team,
                'bet_direction': 'home' if bet_home else 'away',
                'outcome': outcome,
                'profit': stake if outcome == 'win' else -stake,
                'stake': stake,
                'game_info': f"{bet_team} moneyline",
                'actual_score': f"{game.get('home_team_score')}-{game.get('away_team_score')}",
                'confidence': 0.55  # Base home advantage
            })

        return trades

    def _calculate_backtest_results(self, trades: List[Dict], params: Dict) -> Dict[str, Any]:
        """Calculate comprehensive backtest results"""
        if not trades:
            return {'error': 'No trades to analyze'}

        df = pd.DataFrame(trades)

        # Basic metrics
        total_trades = len(df)
        winning_trades = len(df[df['outcome'] == 'win'])
        losing_trades = len(df[df['outcome'] == 'loss'])
        win_rate = winning_trades / total_trades if total_trades > 0 else 0

        # Financial metrics
        total_profit = df['profit'].sum()
        avg_profit = df['profit'].mean()
        avg_win = df[df['outcome'] == 'win']['profit'].mean() if winning_trades > 0 else 0
        avg_loss = df[df['outcome'] == 'loss']['profit'].mean() if losing_trades > 0 else 0

        # Profit factor
        total_wins = df[df['outcome'] == 'win']['profit'].sum()
        total_losses = abs(df[df['outcome'] == 'loss']['profit'].sum())
        profit_factor = total_wins / total_losses if total_losses > 0 else float('inf')

        # Sharpe ratio (simplified)
        returns = df['profit']
        if len(returns) > 1 and returns.std() > 0:
            sharpe_ratio = returns.mean() / returns.std() * np.sqrt(252)  # Annualized
        else:
            sharpe_ratio = 0

        # Maximum drawdown
        cumulative = df['profit'].cumsum()
        running_max = cumulative.expanding().max()
        drawdown = cumulative - running_max
        max_drawdown = drawdown.min()

        # Kelly criterion
        if win_rate > 0 and avg_win > 0 and avg_loss < 0:
            kelly_fraction = win_rate - ((1 - win_rate) / (avg_win / abs(avg_loss)))
            kelly_fraction = max(0, min(kelly_fraction, 0.25))  # Cap at 25%
        else:
            kelly_fraction = 0

        return {
            'total_trades': total_trades,
            'winning_trades': winning_trades,
            'losing_trades': losing_trades,
            'win_rate': round(win_rate * 100, 2),
            'total_profit': round(total_profit, 2),
            'avg_profit_per_trade': round(avg_profit, 2),
            'avg_win': round(avg_win, 2),
            'avg_loss': round(avg_loss, 2),
            'profit_factor': round(profit_factor, 2) if profit_factor != float('inf') else '∞',
            'sharpe_ratio': round(sharpe_ratio, 2),
            'max_drawdown': round(max_drawdown, 2),
            'kelly_fraction': round(kelly_fraction, 4),
            'total_stake': df['stake'].sum(),
            'return_percentage': round((total_profit / df['stake'].sum()) * 100, 2) if df['stake'].sum() > 0 else 0
        }

    def _calculate_risk_metrics(self, trades: List[Dict]) -> Dict[str, Any]:
        """Calculate detailed risk metrics"""
        if not trades:
            return {}

        df = pd.DataFrame(trades)

        # Value at Risk (95% confidence)
        returns = df['profit']
        if len(returns) > 0:
            var_95 = np.percentile(returns, 5)  # 5th percentile = 95% VaR
        else:
            var_95 = 0

        # Expected shortfall (Conditional VaR)
        losses = returns[returns < 0]
        if len(losses) > 0:
            cvar_95 = losses[losses <= var_95].mean() if len(losses[losses <= var_95]) > 0 else var_95
        else:
            cvar_95 = 0

        # Sortino ratio (downside deviation)
        target_return = 0
        downside_returns = returns[returns < target_return]
        if len(downside_returns) > 0:
            downside_deviation = np.sqrt(np.mean(downside_returns ** 2))
            sortino_ratio = returns.mean() / downside_deviation * np.sqrt(252) if downside_deviation > 0 else 0
        else:
            sortino_ratio = 0

        # Calmar ratio (annual return / max drawdown)
        cumulative = df['profit'].cumsum()
        running_max = cumulative.expanding().max()
        drawdown = cumulative - running_max
        max_drawdown = abs(drawdown.min())

        if max_drawdown > 0:
            annual_return = (cumulative.iloc[-1] / len(cumulative)) * 252  # Daily to annual
            calmar_ratio = annual_return / max_drawdown
        else:
            calmar_ratio = 0

        # Win/loss streaks
        streaks = []
        current_streak = 0
        current_type = None

        for outcome in df['outcome']:
            if outcome == current_type:
                current_streak += 1
            else:
                if current_streak > 0:
                    streaks.append({'type': current_type, 'length': current_streak})
                current_streak = 1
                current_type = outcome

        if current_streak > 0:
            streaks.append({'type': current_type, 'length': current_streak})

        win_streaks = [s for s in streaks if s['type'] == 'win']
        loss_streaks = [s for s in streaks if s['type'] == 'loss']

        return {
            'value_at_risk_95': round(var_95, 2),
            'conditional_var_95': round(cvar_95, 2),
            'sortino_ratio': round(sortino_ratio, 2),
            'calmar_ratio': round(calmar_ratio, 2),
            'max_win_streak': max([s['length'] for s in win_streaks]) if win_streaks else 0,
            'max_loss_streak': max([s['length'] for s in loss_streaks]) if loss_streaks else 0,
            'avg_win_streak': round(np.mean([s['length'] for s in win_streaks]), 1) if win_streaks else 0,
            'avg_loss_streak': round(np.mean([s['length'] for s in loss_streaks]), 1) if loss_streaks else 0,
            'volatility': round(df['profit'].std(), 2),
            'skewness': round(df['profit'].skew(), 2),
            'kurtosis': round(df['profit'].kurtosis(), 2)
        }

    def _generate_performance_summary(self, results: Dict[str, Any]) -> str:
        """Generate human-readable performance summary"""
        win_rate = results['win_rate']
        total_profit = results['total_profit']
        profit_factor = results['profit_factor']
        sharpe_ratio = results['sharpe_ratio']
        max_drawdown = results['max_drawdown']

        summary = f"""
🎯 **Backtest Performance Summary**

📊 **Overall Results:**
• Win Rate: {win_rate}%
• Total Profit: ${total_profit}
• Total Trades: {results['total_trades']}

💰 **Profitability Metrics:**
• Profit Factor: {profit_factor}
• Sharpe Ratio: {sharpe_ratio}
• Max Drawdown: ${max_drawdown}

📈 **Risk-Adjusted Returns:**
• Return %: {results['return_percentage']}%
• Kelly Fraction: {results['kelly_fraction']*100}%

🎲 **Trade Analysis:**
• Avg Win: ${results['avg_win']}
• Avg Loss: ${results['avg_loss']}
• Winning Trades: {results['winning_trades']}
• Losing Trades: {results['losing_trades']}
"""

        # Performance rating
        # Handle infinity as a number for comparison
        pf = float('inf') if profit_factor == '∞' else profit_factor
        if win_rate >= 60 and pf >= 1.5:
            rating = "🏆 EXCELLENT - This strategy shows strong potential!"
        elif win_rate >= 55 and pf >= 1.2:
            rating = "✅ GOOD - Solid performance with room for optimization"
        elif win_rate >= 50 and pf >= 1.0:
            rating = "🤔 FAIR - Breakeven performance, needs refinement"
        else:
            rating = "⚠️ POOR - Strategy needs significant improvement"

        summary += f"\n{rating}"

        return summary.strip()

def main():
    """Main function for testing the NBA backtesting engine"""
    import sys

    if len(sys.argv) > 1:
        # API mode - called from Next.js
        engine = NBABacktestingEngine()
        command = sys.argv[1]
        params = json.loads(sys.argv[2]) if len(sys.argv) > 2 else {}

        try:
            if command == "run_backtest":
                result = engine.run_backtest(
                    params['strategy_name'],
                    params.get('parameters', {}),
                    params.get('season', '2023-24'),
                    params.get('min_games', 10)
                )
            else:
                result = {"error": f"Unknown command: {command}"}

            # Output JSON for API consumption
            print(json.dumps(result))

        except Exception as e:
            print(json.dumps({"error": str(e)}))
            sys.exit(1)

    else:
        # Test mode
        engine = NBABacktestingEngine()

        print("🏀 NBA Backtesting Engine Test")
        print("=" * 50)

        # Test player points strategy
        print("\n🏃 Testing Player Points Strategy:")
        result = engine.run_backtest(
            'player_points_over',
            {'player_name': 'lebron', 'points_threshold': 25, 'stake': 100},
            '2023-24',
            5  # Minimum trades for testing
        )

        if result.get('success'):
            print(f"✅ Backtest completed: {result['total_trades']} trades")
            print(f"📊 Win Rate: {result['results']['win_rate']}%")
            print(f"💰 Total Profit: ${result['results']['total_profit']}")
            print(f"📈 Sharpe Ratio: {result['results']['sharpe_ratio']}")
        else:
            print(f"❌ Backtest failed: {result.get('error', 'Unknown error')}")

        # Test team win streak strategy
        print("\n🏆 Testing Team Win Streak Strategy:")
        result2 = engine.run_backtest(
            'team_win_streak',
            {'min_streak': 2, 'stake': 100},
            '2023-24',
            5
        )

        if result2.get('success'):
            print(f"✅ Backtest completed: {result2['total_trades']} trades")
            print(f"📊 Win Rate: {result2['results']['win_rate']}%")
            print(f"💰 Total Profit: ${result2['results']['total_profit']}")
        else:
            print(f"❌ Backtest failed: {result2.get('error', 'Unknown error')}")

        print("\n✅ NBA Backtesting engine test completed!")

if __name__ == "__main__":
    main()
