/**
 * Jest Setup Configuration
 * Global test setup for React Testing Library and custom matchers
 */

// Note: @testing-library/jest-dom requires React testing environment
// For now, we'll skip it since we're testing Node.js code

// Mock Next.js router
jest.mock('next/navigation', () => ({
  useRouter: () => ({
    push: jest.fn(),
    replace: jest.fn(),
    back: jest.fn(),
    forward: jest.fn(),
    refresh: jest.fn(),
    prefetch: jest.fn(),
  }),
  useSearchParams: () => new URLSearchParams(),
  usePathname: () => '/',
}));

// Mock environment variables
process.env.NEXT_PUBLIC_BASE_URL = 'http://localhost:3000';
process.env.JWT_SECRET = 'test-jwt-secret';
process.env.DATABASE_URL = 'mysql://test:test@localhost:3306/test_db';

// Global test utilities
global.testUtils = {
  // Create mock user
  createMockUser: (overrides = {}) => ({
    id: 'user_123',
    email: 'test@example.com',
    firstName: 'John',
    lastName: 'Doe',
    subscriptionTier: 'FREE',
    emailVerified: true,
    ...overrides,
  }),

  // Create mock strategy
  createMockStrategy: (overrides = {}) => ({
    id: 'strategy_123',
    name: 'Test Strategy',
    description: 'A test strategy',
    domain: 'STOCKS',
    rules: { trigger: 'momentum', threshold: 0.05 },
    isPublic: false,
    userId: 'user_123',
    ...overrides,
  }),

  // Mock fetch responses
  mockFetchResponse: (data, status = 200) => ({
    ok: status >= 200 && status < 300,
    status,
    json: () => Promise.resolve(data),
    text: () => Promise.resolve(JSON.stringify(data)),
  }),

  // Wait for component updates
  waitForNextTick: () => new Promise(resolve => setTimeout(resolve, 0)),
};

// Mock database for tests
jest.mock('@/lib/database', () => ({
  DatabaseService: {
    createUser: jest.fn(),
    findUserByEmail: jest.fn(),
    findUserById: jest.fn(),
    updateUser: jest.fn(),
    verifyPassword: jest.fn(),
    generateAuthToken: jest.fn(),
    createStrategy: jest.fn(),
    findStrategyById: jest.fn(),
    updateStrategy: jest.fn(),
    hashPassword: jest.fn(),
    generateVerificationToken: jest.fn(),
    findUserByVerificationToken: jest.fn(),
    recordUserEvent: jest.fn(),
  },
}));

// Mock Stripe
jest.mock('@/lib/stripe', () => ({
  StripeService: {
    createCustomer: jest.fn(),
    createSubscription: jest.fn(),
    updateSubscription: jest.fn(),
    cancelSubscription: jest.fn(),
  },
  SUBSCRIPTION_PLANS: {
    FREE: { name: 'Free Trial', price: 0 },
    PRO: { name: 'Pro', price: 20 },
    ELITE: { name: 'Elite', price: 200 },
  },
}));

// Mock email service
jest.mock('@/lib/email', () => ({
  EmailService: {
    sendEmail: jest.fn(),
  },
  EmailUtils: {
    sendWelcomeEmail: jest.fn(),
    sendEmailVerification: jest.fn(),
    sendPasswordReset: jest.fn(),
  },
}));

// Suppress console errors during tests unless explicitly testing for them
const originalError = console.error;
beforeAll(() => {
  console.error = (...args) => {
    if (
      typeof args[0] === 'string' &&
      args[0].includes('Warning:') &&
      args[0].includes('ReactDOMTestUtils')
    ) {
      return;
    }
    originalError.call(console, ...args);
  };
});

afterAll(() => {
  console.error = originalError;
});
