#!/usr/bin/env python3
"""
SportsData.io API Integration Report Generator
For EvenTheOdds.ai
"""

import json
import os
from datetime import datetime

def generate_report():
    print("=" * 70)
    print("SPORTSDATA.IO API INTEGRATION REPORT")
    print("EvenTheOdds.ai - Live Server Analysis")
    print(f"Generated: {datetime.now().strftime('%Y-%m-%d %H:%M:%S UTC')}")
    print("=" * 70)

    cache_dir = "/var/www/html/eventheodds/data/nba_cache"
    sportsdata_cache = "/var/www/html/eventheodds/data/sportsdata_cache"
    env_file = "/var/www/html/eventheodds/.env"

    # Section 1: Cache Status
    print("\n## 1. CACHE STATUS")
    print("-" * 40)

    if os.path.exists(cache_dir):
        files = os.listdir(cache_dir)
        total_size = sum(os.path.getsize(os.path.join(cache_dir, f)) for f in files if f.endswith(".json"))
        print(f"NBA Cache Directory: {cache_dir}")
        print(f"  Files: {len(files)}")
        print(f"  Total Size: {total_size / 1024 / 1024:.2f} MB")
        
        meta_path = os.path.join(cache_dir, "cache_metadata.json")
        if os.path.exists(meta_path):
            with open(meta_path) as f:
                meta = json.load(f)
                print("  Cached Endpoints:")
                for key, info in meta.items():
                    endpoint = info.get("metadata", {}).get("endpoint", "unknown")
                    ts = info.get("timestamp", "?")[:19]
                    size_kb = info.get("size", 0) / 1024
                    print(f"    - {endpoint}: {size_kb:.1f} KB (cached: {ts})")

    if os.path.exists(sportsdata_cache):
        sd_files = [f for f in os.listdir(sportsdata_cache) if f.endswith(".json")]
        print(f"\nSportsData.io TypeScript Cache: {len(sd_files)} files")
    else:
        print("\nSportsData.io TypeScript Cache: Not initialized")

    # Section 2: Data Analysis
    print("\n## 2. CACHED DATA ANALYSIS")
    print("-" * 40)

    games_file = os.path.join(cache_dir, "e5710eaa6f62fc23566fe952fe4930dd.json")
    games = []
    if os.path.exists(games_file):
        with open(games_file) as f:
            games = json.load(f)
        
        print("NBA Games Dataset:")
        print(f"  Total Games: {len(games)}")
        if games:
            first_game = games[0]
            print(f"  Season: {first_game.get('season', 'N/A')}")
            print(f"  Season Type: {first_game.get('season_type', 'N/A')}")
        
        completed = sum(1 for g in games if g.get("home_team_score") is not None)
        pct = 100 * completed / len(games) if games else 0
        print(f"  Games with scores: {completed}/{len(games)} ({pct:.1f}%)")
        
        dates = sorted([g.get("game_date", "") for g in games if g.get("game_date")])
        if dates:
            print(f"  Date Range: {dates[0]} to {dates[-1]}")

    teams_file = os.path.join(cache_dir, "102cd652129ece8cfcd1747c333ef64a.json")
    if os.path.exists(teams_file):
        with open(teams_file) as f:
            teams = json.load(f)
        print(f"\nNBA Teams: {len(teams)} teams cached")

    players_file = os.path.join(cache_dir, "9e9d129ad0d2a5675fe3654c614b9d40.json")
    if os.path.exists(players_file):
        with open(players_file) as f:
            players = json.load(f)
        print(f"NBA Players: {len(players)} players cached")

    # Section 3: API Configuration
    print("\n## 3. API CONFIGURATION")
    print("-" * 40)

    env_content = ""
    if os.path.exists(env_file):
        with open(env_file) as f:
            env_content = f.read()
    
    keys_to_check = [
        ("SPORTSDATA_API_KEY", "SportsData.io API access"),
        ("GROK_API_KEY", "Grok AI for strategy generation"),
        ("STRIPE_SECRET_KEY", "Payment processing"),
        ("JWT_SECRET", "Authentication tokens"),
    ]
    
    for key, desc in keys_to_check:
        present = key in env_content
        symbol = "✅" if present else "❌"
        status = "CONFIGURED" if present else "MISSING"
        print(f"  {symbol} {key}: {status}")
        if not present:
            print(f"      └─ Required for: {desc}")

    # Section 4: Sample Data
    print("\n## 4. SAMPLE CACHED GAMES")
    print("-" * 40)
    if games:
        print("Recent games from cache:")
        for g in games[:5]:
            home = g.get("home_team_name", "?")
            away = g.get("away_team_id", "?")
            date = g.get("game_date", "?")
            score_h = g.get("home_team_score", "-")
            score_a = g.get("away_team_score", "-")
            print(f"  {date}: {away} @ {home} | Score: {score_a}-{score_h}")

    # Section 5: Issues
    print("\n## 5. ISSUES IDENTIFIED")
    print("-" * 40)
    
    issues = []
    if "SPORTSDATA_API_KEY" not in env_content:
        issues.append(("🚨 CRITICAL", "SPORTSDATA_API_KEY not configured", 
                       "Cannot fetch live sports data from SportsData.io API"))
    
    if games:
        null_away = sum(1 for g in games if g.get("away_team_score") is None)
        if null_away > 0:
            issues.append(("⚠️ DATA", f"{null_away} games missing away_team_score",
                          "Backtest accuracy may be affected"))
    
    issues.append(("⚠️ CACHE", "Data cached on Nov 28, 2025",
                   "May not include 2024-25 season games"))
    
    for priority, issue, impact in issues:
        print(f"\n  {priority}: {issue}")
        print(f"      Impact: {impact}")

    # Section 6: Recommendations
    print("\n## 6. RECOMMENDATIONS")
    print("-" * 40)
    recommendations = [
        "Add SPORTSDATA_API_KEY to .env file for live data access",
        "Subscribe to SportsData.io API at https://sportsdata.io/",
        "Refresh NBA cache with 2024-25 season data",
        "TypeScript caching layer is ready - will auto-cache API calls",
        "Cache TTLs configured: Teams=7 days, Scores=5 min, Historical=30 days",
    ]
    for i, rec in enumerate(recommendations, 1):
        print(f"  {i}. {rec}")

    # Section 7: API Endpoints Available
    print("\n## 7. AVAILABLE API ENDPOINTS")
    print("-" * 40)
    print("  GET /api/nba/teams          - NBA Teams list")
    print("  GET /api/nba/players        - NBA Players list")
    print("  GET /api/nba/games          - NBA Games (requires season param)")
    print("  GET /api/sportsdata         - Direct SportsData.io query (auth required)")
    print("  POST /api/chat              - AI Strategy Generation (auth required)")

    print("\n" + "=" * 70)
    print("END OF REPORT")
    print("=" * 70)

if __name__ == "__main__":
    generate_report()
