/**
 * Strategy Creation and Simulation E2E Tests
 * Tests the core trading strategy functionality
 */

import { test, expect, Page } from '@playwright/test';

test.describe('Strategy Flow', () => {
  let authToken: string;
  let testUser = {
    email: process.env.TEST_USER_EMAIL || 'test@example.com',
    password: process.env.TEST_USER_PASSWORD || 'password123'
  };

  test.beforeEach(async ({ page }) => {
    // Try to authenticate for strategy tests
    console.log('🔐 Attempting to authenticate for strategy tests');

    await page.goto('/login');
    await page.fill('input[name="email"]', testUser.email);
    await page.fill('input[name="password"]', testUser.password);

    try {
      await page.click('button[type="submit"]');
      await page.waitForURL('**/dashboard', { timeout: 5000 });
      console.log('✅ Authentication successful');
    } catch (error) {
      console.log('⚠️  Authentication failed - tests will run in unauthenticated mode');
    }
  });

  test('Strategy playground access', async ({ page }) => {
    console.log('🧪 Testing: Strategy Playground Access');

    await page.goto('/');
    console.log('📄 Loaded homepage');

    // Try to access strategy playground
    const playgroundLink = page.locator('text=Strategy Playground, a[href*="playground"]').first();

    if (await playgroundLink.isVisible()) {
      await playgroundLink.click();
      console.log('🎯 Clicked strategy playground link');

      await page.waitForTimeout(1000);
      const currentUrl = page.url();

      if (currentUrl.includes('playground') || currentUrl.includes('strategy')) {
        console.log('✅ Strategy playground accessible');
      } else {
        console.log('⚠️  Strategy playground redirect behavior unclear');
      }
    } else {
      console.log('⚠️  Strategy playground link not found on homepage');
    }
  });

  test('Strategy creation form', async ({ page }) => {
    console.log('🧪 Testing: Strategy Creation Form');

    // Navigate to strategy creation (try multiple paths)
    const strategyPaths = ['/playground', '/strategies', '/dashboard'];
    let foundStrategyPage = false;

    for (const path of strategyPaths) {
      try {
        await page.goto(path);
        console.log(`📄 Trying path: ${path}`);

        // Look for strategy creation elements
        const strategyForm = await page.locator('textarea, input[placeholder*="strategy"], [data-testid*="strategy"]').first();

        if (await strategyForm.isVisible()) {
          console.log(`✅ Found strategy form at: ${path}`);
          foundStrategyPage = true;
          break;
        }
      } catch (error) {
        console.log(`⚠️  Path ${path} failed:`, error.message);
      }
    }

    if (!foundStrategyPage) {
      console.log('⚠️  Strategy creation page not found - checking current page elements');

      // Check what's actually on the current page
      const pageTitle = await page.title();
      const bodyText = await page.textContent('body');

      console.log('📄 Current page title:', pageTitle);
      console.log('📄 Page content preview:', bodyText?.substring(0, 300));
      return;
    }

    // Test different strategy prompts
    const testPrompts = [
      'Create a momentum strategy for stocks when price increases by 5%',
      'Buy sports teams after 3 consecutive losses',
      'Create a parlay strategy with $10 per leg for crypto',
      'Forex breakout strategy when price breaks resistance'
    ];

    for (const prompt of testPrompts) {
      console.log(`🎯 Testing prompt: "${prompt.substring(0, 50)}..."`);

      try {
        // Clear and fill the strategy input
        const inputField = page.locator('textarea, input[type="text"]').first();
        await inputField.clear();
        await inputField.fill(prompt);

        // Select domain if dropdown exists
        const domainSelect = page.locator('select, [role="combobox"]').first();
        if (await domainSelect.isVisible()) {
          const domain = prompt.toLowerCase().includes('stock') ? 'stocks' :
                        prompt.toLowerCase().includes('sport') ? 'sports' :
                        prompt.toLowerCase().includes('crypto') ? 'crypto' : 'forex';
          await domainSelect.selectOption(domain);
          console.log(`🏷️  Selected domain: ${domain}`);
        }

        // Submit strategy
        const submitButton = page.locator('button[type="submit"], button:has-text("Create"), button:has-text("Run")').first();
        await submitButton.click();

        console.log('🚀 Submitted strategy creation request');

        // Wait for response
        await page.waitForTimeout(3000);

        // Check for results or errors
        const successIndicators = [
          'text=Strategy created',
          'text=Simulation complete',
          'text=Results',
          '.chart', // Chart elements
          '[data-testid*="result"]'
        ];

        let strategySuccess = false;
        for (const indicator of successIndicators) {
          if (await page.locator(indicator).isVisible()) {
            console.log(`✅ Strategy simulation successful - found: ${indicator}`);
            strategySuccess = true;
            break;
          }
        }

        if (!strategySuccess) {
          // Check for error messages
          const errorMessages = await page.locator('.text-red-500, .text-red-600, .alert-error').allTextContents();
          if (errorMessages.length > 0) {
            console.log('❌ Strategy creation failed:', errorMessages.join(', '));
          } else {
            console.log('⚠️  Strategy result unclear - no success or error indicators found');
          }
        }

        // Check for API calls
        const responses = [];
        page.on('response', response => {
          if (response.url().includes('/api/strategies')) {
            responses.push({
              url: response.url(),
              status: response.status(),
              ok: response.ok()
            });
          }
        });

        await page.waitForTimeout(1000);

        if (responses.length > 0) {
          console.log('🔗 API calls made:', responses);
        }

      } catch (error) {
        console.log(`❌ Error testing prompt "${prompt.substring(0, 30)}...":`, error.message);
      }
    }
  });

  test('Strategy results visualization', async ({ page }) => {
    console.log('🧪 Testing: Strategy Results Visualization');

    // This test assumes we have a strategy created
    // Look for charts, tables, or result displays
    const chartElements = [
      'canvas', // Chart.js canvases
      '.recharts-wrapper', // Recharts
      '[data-testid*="chart"]',
      '.chart',
      'svg', // SVG charts
    ];

    const tableElements = [
      'table',
      '.table',
      '[data-testid*="table"]',
      'tbody tr', // Table rows
    ];

    let foundCharts = false;
    let foundTables = false;

    for (const selector of chartElements) {
      const count = await page.locator(selector).count();
      if (count > 0) {
        console.log(`📊 Found ${count} chart elements: ${selector}`);
        foundCharts = true;
      }
    }

    for (const selector of tableElements) {
      const count = await page.locator(selector).count();
      if (count > 0) {
        console.log(`📋 Found ${count} table elements: ${selector}`);
        foundTables = true;
      }
    }

    if (!foundCharts && !foundTables) {
      console.log('⚠️  No charts or tables found for results visualization');

      // Check for any numerical results or metrics
      const numberElements = await page.locator('text=/\\d+\\.\\d+|\\d+\\%/').allTextContents();
      if (numberElements.length > 0) {
        console.log('🔢 Found numerical results:', numberElements.slice(0, 5));
      } else {
        console.log('⚠️  No numerical results found either');
      }
    } else {
      console.log('✅ Results visualization elements found');
    }
  });

  test('Strategy library access', async ({ page }) => {
    console.log('🧪 Testing: Strategy Library Access');

    // Try to access strategy library
    const libraryPaths = ['/strategies', '/library', '/dashboard'];

    for (const path of libraryPaths) {
      try {
        await page.goto(path);
        console.log(`📚 Checking strategy library at: ${path}`);

        // Look for strategy list elements
        const strategyItems = await page.locator('.strategy-item, .card, [data-testid*="strategy"]').all();

        if (strategyItems.length > 0) {
          console.log(`✅ Found ${strategyItems.length} strategy items in library`);
          break;
        }

        // Check for empty state
        const emptyState = await page.locator('text=No strategies, text=Empty, text=Create your first').isVisible();
        if (emptyState) {
          console.log('📭 Strategy library is empty (expected for new users)');
          break;
        }

      } catch (error) {
        console.log(`⚠️  Library path ${path} failed:`, error.message);
      }
    }
  });

  test('API response monitoring', async ({ page }) => {
    console.log('🧪 Testing: API Response Monitoring');

    const apiCalls: any[] = [];

    // Monitor all API calls
    page.on('request', request => {
      if (request.url().includes('/api/')) {
        apiCalls.push({
          method: request.method(),
          url: request.url(),
          timestamp: Date.now()
        });
      }
    });

    page.on('response', response => {
      if (response.url().includes('/api/')) {
        const matchingRequest = apiCalls.find(call =>
          call.url === response.url() &&
          call.method === response.request().method()
        );

        if (matchingRequest) {
          matchingRequest.status = response.status();
          matchingRequest.duration = Date.now() - matchingRequest.timestamp;
          matchingRequest.ok = response.ok();
        }
      }
    });

    // Navigate through the app to trigger API calls
    await page.goto('/');
    await page.waitForTimeout(1000);

    // Try strategy-related actions
    const strategyButton = page.locator('button, a').filter({ hasText: /strategy|create|run/i }).first();
    if (await strategyButton.isVisible()) {
      await strategyButton.click();
      await page.waitForTimeout(2000);
    }

    // Log all API calls made
    console.log('🔗 API calls during session:');
    apiCalls.forEach(call => {
      const status = call.status ? `${call.status} (${call.ok ? 'OK' : 'ERROR'})` : 'PENDING';
      const duration = call.duration ? `${call.duration}ms` : 'UNKNOWN';
      console.log(`  ${call.method} ${call.url} - ${status} - ${duration}`);
    });

    if (apiCalls.length === 0) {
      console.log('⚠️  No API calls detected - application may not be fully functional');
    }
  });
});
