import { test, expect } from '@playwright/test';

test('Dashboard loads without hydration errors', async ({ page }) => {
  // Listen for hydration errors
  const hydrationErrors: string[] = [];
  const consoleErrors: string[] = [];

  page.on('console', msg => {
    if (msg.type() === 'error') {
      consoleErrors.push(msg.text());
      if (msg.text().includes('hydration') ||
          msg.text().includes('Hydration') ||
          msg.text().includes('Text content did not match') ||
          msg.text().includes('does not match server-rendered HTML')) {
        hydrationErrors.push(msg.text());
      }
    }
  });

  // Navigate to dashboard
  await page.goto('http://localhost:3003/dashboard');

  // Wait for page to fully load
  await page.waitForLoadState('networkidle');
  await page.waitForTimeout(2000); // Extra time for hydration to complete

  // Check that no hydration errors occurred
  console.log('Hydration errors found:', hydrationErrors.length);
  console.log('Console errors found:', consoleErrors.length);

  // Filter out non-critical errors
  const criticalErrors = hydrationErrors.filter(error =>
    !error.includes('Download the React DevTools') &&
    !error.includes('Image with src') &&
    !error.includes('Warning:') &&
    error.includes('hydration') || error.includes('Hydration') ||
    error.includes('Text content did not match') ||
    error.includes('does not match server-rendered HTML')
  );

  // Should have no critical hydration errors
  expect(criticalErrors.length).toBe(0);

  // Verify the page is functional
  await expect(page.locator('text=AI Strategy Assistant')).toBeVisible();
  await expect(page.locator('text=AI Chat')).toBeVisible();

  // Check that timestamps are displayed (without hydration errors)
  const timestampElements = page.locator('p[class*="text-xs"][class*="opacity-60"]');
  if (await timestampElements.count() > 0) {
    await expect(timestampElements.first()).toBeVisible();
  }
});
