import { test, expect } from '@playwright/test';

test.describe('Homepage User Flow', () => {
  test('should load homepage with proper branding and styling', async ({ page }) => {
    // Navigate to homepage
    await page.goto('http://localhost:3003/');

    // Check page loads
    await expect(page).toHaveTitle(/QuantEdge AI/);

    // Check branding in header
    await expect(page.locator('header').locator('text=QuantEdge AI')).toBeVisible();

    // Check main heading
    await expect(page.locator('text=Backtest Strategies with Conversational AI')).toBeVisible();

    // Check glassmorphism styling is applied (glass-panel class should exist)
    const glassPanels = page.locator('.glass-panel');
    await expect(glassPanels.first()).toBeVisible();

    // Check CTA buttons
    await expect(page.locator('text=Start Backtesting')).toBeVisible();
    await expect(page.locator('text=View Demo')).toBeVisible();

    // Check navigation links in header
    const nav = page.locator('nav');
    await expect(nav.locator('text=Dashboard')).toBeVisible();
    await expect(nav.locator('text=Strategies')).toBeVisible();

    // Check auth buttons
    await expect(page.locator('text=Sign In')).toBeVisible();
    await expect(page.locator('text=Get Started')).toBeVisible();
  });

  test('should navigate to dashboard', async ({ page }) => {
    await page.goto('http://localhost:3003/');

    // Click dashboard link
    await page.locator('text=Dashboard').click();

    // Should redirect to login or dashboard
    await page.waitForURL(/\/login|\/dashboard/);
  });

  test('should navigate to help center', async ({ page }) => {
    await page.goto('http://localhost:3003/');

    // Navigate to help page directly
    await page.goto('http://localhost:3003/help');

    await expect(page.locator('text=Help Center')).toBeVisible();
    await expect(page.locator('text=Frequently Asked Questions')).toBeVisible();
  });

  test('should navigate to community forum', async ({ page }) => {
    await page.goto('http://localhost:3003/');

    // Navigate to community page directly
    await page.goto('http://localhost:3003/community');

    await expect(page.locator('text=Community Forum')).toBeVisible();
    await expect(page.locator('text=Connect with traders')).toBeVisible();
  });

  test('should navigate to support page', async ({ page }) => {
    await page.goto('http://localhost:3003/');

    // Navigate to support page directly
    await page.goto('http://localhost:3003/support');

    await expect(page.locator('text=Contact Support')).toBeVisible();
    await expect(page.locator('text=Send us a Message')).toBeVisible();
  });

  test('should check responsive design', async ({ page }) => {
    await page.goto('http://localhost:3003/');

    // Test mobile viewport
    await page.setViewportSize({ width: 375, height: 667 });

    // Check hamburger menu appears on mobile (hidden on desktop)
    const navMenu = page.locator('nav').first();
    await expect(navMenu).toHaveClass(/hidden md:flex/);

    // Reset to desktop
    await page.setViewportSize({ width: 1200, height: 800 });

    // Check nav is visible on desktop (has md:flex class)
    await expect(navMenu).toHaveClass(/md:flex/);
  });

  test('should test announcements API', async ({ page }) => {
    const response = await page.request.get('http://localhost:3003/api/announcements');
    expect(response.status()).toBe(200);

    const data = await response.json();
    expect(data).toHaveProperty('announcements');
    expect(data).toHaveProperty('total');
    expect(Array.isArray(data.announcements)).toBe(true);
  });

  test('should test feedback form submission', async ({ page }) => {
    await page.goto('http://localhost:3003/dashboard');

    // Click feedback tab (assuming user is logged in or we mock auth)
    const feedbackTab = page.locator('text=Feedback');
    if (await feedbackTab.isVisible()) {
      await feedbackTab.click();

      // Check feedback form elements
      await expect(page.locator('text=Submit Feedback')).toBeVisible();
      await expect(page.locator('text=Type of Feedback')).toBeVisible();
      await expect(page.locator('text=Title')).toBeVisible();
      await expect(page.locator('text=Description')).toBeVisible();
    }
  });
});
