/**
 * Chat and Admin Functionality E2E Tests
 * Tests AI chat interactions and admin dashboard features
 */

import { test, expect, Page } from '@playwright/test';

test.describe('Chat and Admin Flow', () => {
  let authToken: string;
  let isAdmin = false;

  test.beforeEach(async ({ page }) => {
    // Try to authenticate
    console.log('🔐 Setting up authentication for chat/admin tests');

    const testUser = {
      email: process.env.TEST_USER_EMAIL || 'admin@example.com',
      password: process.env.TEST_USER_PASSWORD || 'password123'
    };

    try {
      await page.goto('/login');
      await page.fill('input[name="email"]', testUser.email);
      await page.fill('input[name="password"]', testUser.password);
      await page.click('button[type="submit"]');

      // Check if login successful
      await page.waitForTimeout(2000);
      const currentUrl = page.url();

      if (currentUrl.includes('/admin')) {
        isAdmin = true;
        console.log('👑 Admin user authenticated');
      } else if (currentUrl.includes('/dashboard') || currentUrl.includes('/welcome')) {
        console.log('✅ Regular user authenticated');
      } else {
        console.log('⚠️  Authentication result unclear');
      }
    } catch (error) {
      console.log('⚠️  Authentication failed - tests will run in unauthenticated mode');
    }
  });

  test('Chat interface access and basic functionality', async ({ page }) => {
    console.log('🧪 Testing: Chat Interface Access');

    // Try to access chat from various locations
    const chatAccessPoints = [
      () => page.locator('text=Chat with AI, button:has-text("Chat")').first(),
      () => page.locator('a[href*="chat"]').first(),
      () => page.goto('/chat')
    ];

    let chatAccessible = false;

    for (let i = 0; i < chatAccessPoints.length; i++) {
      try {
        const accessPoint = chatAccessPoints[i];

        if (typeof accessPoint === 'function' && accessPoint.length === 0) {
          // It's a locator function
          const element = accessPoint();
          if (await element.isVisible()) {
            await element.click();
            console.log(`💬 Accessed chat via UI element ${i + 1}`);
            chatAccessible = true;
            break;
          }
        } else {
          // It's a navigation function
          await (accessPoint as any)();
          console.log(`💬 Navigated to chat via direct URL`);
          chatAccessible = true;
          break;
        }
      } catch (error) {
        console.log(`⚠️  Chat access method ${i + 1} failed:`, error.message);
      }
    }

    if (!chatAccessible) {
      console.log('⚠️  Chat interface not accessible - checking current page');

      // Check what's on the current page
      const pageTitle = await page.title();
      const chatElements = await page.locator('textarea, input[placeholder*="message"], [data-testid*="chat"]').count();

      console.log(`📄 Current page: ${pageTitle}`);
      console.log(`💬 Found ${chatElements} potential chat elements`);

      if (chatElements === 0) {
        console.log('❌ No chat interface found');
        return;
      }
    }

    // Test chat input
    const chatInput = page.locator('textarea, input[type="text"]').filter({
      hasText: '',
      hasAttribute: 'placeholder'
    }).first();

    if (await chatInput.isVisible()) {
      console.log('✅ Chat input field found');

      // Test different chat messages
      const testMessages = [
        'Create a simple momentum strategy for stocks',
        'What are some good trading strategies for crypto?',
        'Explain the difference between a parlay and round robin bet'
      ];

      for (const message of testMessages) {
        console.log(`💬 Testing chat message: "${message.substring(0, 40)}..."`);

        try {
          // Clear and type message
          await chatInput.clear();
          await chatInput.fill(message);

          // Find and click send button
          const sendButton = page.locator('button[type="submit"], button:has-text("Send"), [data-testid*="send"]').first();
          await sendButton.click();

          console.log('🚀 Sent chat message');

          // Wait for response
          await page.waitForTimeout(3000);

          // Check for response indicators
          const responseIndicators = [
            '.message-response',
            '.ai-response',
            '[data-testid*="response"]',
            'text=AI:', // AI response prefix
            'text=function', // Strategy code response
          ];

          let responseFound = false;
          for (const indicator of responseIndicators) {
            if (await page.locator(indicator).isVisible()) {
              console.log(`✅ Chat response received - found: ${indicator}`);
              responseFound = true;
              break;
            }
          }

          if (!responseFound) {
            // Check for loading indicators
            const loadingElements = await page.locator('.loading, .spinner, text=Thinking, text=Generating').count();
            if (loadingElements > 0) {
              console.log('⏳ Chat response in progress (loading indicators found)');
            } else {
              console.log('⚠️  No chat response or loading indicators found');
            }
          }

          // Check for API calls
          const chatApiCalls = [];
          page.on('response', response => {
            if (response.url().includes('/api/chat')) {
              chatApiCalls.push({
                url: response.url(),
                status: response.status(),
                ok: response.ok()
              });
            }
          });

          await page.waitForTimeout(1000);

          if (chatApiCalls.length > 0) {
            console.log('🔗 Chat API calls:', chatApiCalls);
          }

        } catch (error) {
          console.log(`❌ Error testing chat message:`, error.message);
        }
      }
    } else {
      console.log('❌ Chat input field not found');
    }
  });

  test('Admin dashboard access and functionality', async ({ page }) => {
    console.log('🧪 Testing: Admin Dashboard Access');

    if (!isAdmin) {
      console.log('⚠️  Not authenticated as admin - attempting direct access');

      // Try direct navigation
      await page.goto('/admin');
      console.log('📄 Attempted direct navigation to admin');

      await page.waitForTimeout(2000);
      const currentUrl = page.url();

      if (currentUrl.includes('/admin')) {
        console.log('✅ Admin dashboard accessible');
      } else if (currentUrl.includes('/login')) {
        console.log('🔒 Admin correctly protected - redirected to login');
        return;
      } else {
        console.log('⚠️  Admin access behavior unclear');
        return;
      }
    }

    // Test admin dashboard sections
    const adminSections = [
      { name: 'Overview', selector: 'text=Overview, text=Dashboard' },
      { name: 'Data Management', selector: 'text=Data, text=Datasets' },
      { name: 'Strategies', selector: 'text=Strategies, text=Strategy' },
      { name: 'Analytics', selector: 'text=Analytics, text=Stats' },
      { name: 'Settings', selector: 'text=Settings, text=Configuration' }
    ];

    for (const section of adminSections) {
      try {
        const sectionElement = page.locator(section.selector).first();

        if (await sectionElement.isVisible()) {
          console.log(`✅ Admin section "${section.name}" accessible`);
        } else {
          console.log(`⚠️  Admin section "${section.name}" not found`);
        }
      } catch (error) {
        console.log(`❌ Error checking admin section "${section.name}":`, error.message);
      }
    }

    // Test admin API endpoints
    console.log('🔗 Testing admin API endpoints');

    const adminApiCalls = [];
    page.on('response', response => {
      if (response.url().includes('/api/admin')) {
        adminApiCalls.push({
          url: response.url(),
          status: response.status(),
          ok: response.ok()
        });
      }
    });

    // Navigate through admin sections to trigger API calls
    const adminLinks = page.locator('a[href*="/admin"]').all();
    for (const link of await adminLinks) {
      try {
        await link.click();
        await page.waitForTimeout(1000);
      } catch (error) {
        // Ignore navigation errors
      }
    }

    console.log('🔗 Admin API calls made:', adminApiCalls);

    if (adminApiCalls.length === 0) {
      console.log('⚠️  No admin API calls detected');
    }
  });

  test('Performance monitoring and caching', async ({ page }) => {
    console.log('🧪 Testing: Performance Monitoring and Caching');

    const performanceMetrics: any[] = [];
    const apiCalls: any[] = [];

    // Monitor performance
    page.on('response', response => {
      const timing = response.request().timing();
      if (response.url().includes('/api/')) {
        apiCalls.push({
          url: response.url(),
          status: response.status(),
          duration: timing ? timing.receiveHeadersEnd - timing.sendStart : 0,
          cached: response.headers()['x-cache-status'] === 'HIT'
        });
      }
    });

    // Navigate through key pages
    const testPages = ['/', '/dashboard', '/strategies', '/admin'];

    for (const pagePath of testPages) {
      try {
        console.log(`📊 Testing performance for: ${pagePath}`);

        const startTime = Date.now();
        await page.goto(pagePath);
        const loadTime = Date.now() - startTime;

        console.log(`⏱️  Page load time: ${loadTime}ms`);

        // Check for performance headers
        const responseTime = await page.evaluate(() => {
          const perfEntries = performance.getEntriesByType('navigation');
          return perfEntries.length > 0 ? perfEntries[0].duration : 0;
        });

        console.log(`⏱️  Browser-reported load time: ${responseTime}ms`);

        // Look for caching indicators
        const cacheHeaders = await page.evaluate(() => {
          const resources = performance.getEntriesByType('resource');
          return resources.map(r => ({
            url: r.name,
            duration: r.duration,
            cached: (r as any).transferSize === 0
          })).filter(r => r.url.includes('/api/')).slice(0, 5);
        });

        if (cacheHeaders.length > 0) {
          console.log('💾 Cache status for API calls:', cacheHeaders);
        }

        await page.waitForTimeout(1000); // Let API calls complete

      } catch (error) {
        console.log(`❌ Error testing page ${pagePath}:`, error.message);
      }
    }

    // Summary of API performance
    console.log('📊 API Performance Summary:');
    apiCalls.forEach(call => {
      const cacheStatus = call.cached ? ' (CACHED)' : '';
      console.log(`  ${call.url} - ${call.status} - ${call.duration}ms${cacheStatus}`);
    });

    // Check for slow requests
    const slowRequests = apiCalls.filter(call => call.duration > 1000);
    if (slowRequests.length > 0) {
      console.log('🐌 Slow API requests (>1s):', slowRequests);
    }

    // Check cache hit rate
    const cachedRequests = apiCalls.filter(call => call.cached).length;
    const cacheHitRate = apiCalls.length > 0 ? (cachedRequests / apiCalls.length * 100).toFixed(1) : 0;
    console.log(`💾 Cache hit rate: ${cacheHitRate}% (${cachedRequests}/${apiCalls.length})`);
  });

  test('Error handling and edge cases', async ({ page }) => {
    console.log('🧪 Testing: Error Handling and Edge Cases');

    // Test invalid URLs
    const invalidUrls = ['/nonexistent', '/api/invalid-endpoint', '/admin/secret-area'];

    for (const url of invalidUrls) {
      try {
        const response = await page.goto(url);
        const statusCode = response?.status();

        if (statusCode === 404) {
          console.log(`✅ ${url} correctly returns 404`);
        } else if (statusCode === 200) {
          console.log(`⚠️  ${url} unexpectedly returns 200 - may need protection`);
        } else {
          console.log(`❓ ${url} returns status: ${statusCode}`);
        }
      } catch (error) {
        console.log(`❌ Error accessing ${url}:`, error.message);
      }
    }

    // Test form validation
    await page.goto('/signup');

    // Try to submit empty form
    const submitButton = page.locator('button[type="submit"]').first();
    if (await submitButton.isVisible()) {
      await submitButton.click();

      await page.waitForTimeout(1000);

      // Check for validation errors
      const errorMessages = await page.locator('.text-red-500, .text-red-600, [role="alert"]').allTextContents();

      if (errorMessages.length > 0) {
        console.log('✅ Form validation working - found error messages:', errorMessages.slice(0, 3));
      } else {
        console.log('⚠️  Form validation may not be working - no error messages found');
      }
    }

    // Test network error simulation (if supported)
    try {
      // This would require custom network interception
      console.log('⚠️  Network error testing requires custom setup');
    } catch (error) {
      console.log('❌ Network error test failed:', error.message);
    }
  });

  test('Cross-browser and responsive behavior', async ({ page, browserName }) => {
    console.log(`🧪 Testing: Cross-browser behavior (${browserName})`);

    // Test responsive breakpoints
    const viewports = [
      { width: 1920, height: 1080, name: 'Desktop' },
      { width: 768, height: 1024, name: 'Tablet' },
      { width: 375, height: 667, name: 'Mobile' }
    ];

    for (const viewport of viewports) {
      await page.setViewportSize({ width: viewport.width, height: viewport.height });
      console.log(`📱 Testing ${viewport.name} viewport (${viewport.width}x${viewport.height})`);

      // Reload page to test responsive behavior
      await page.reload();

      // Check for mobile menu or responsive elements
      const mobileMenu = await page.locator('.mobile-menu, [data-testid*="mobile"], button[aria-label*="menu"]').count();
      const hiddenElements = await page.locator('.hidden.md\\:block, .block.md\\:hidden').count();

      if (viewport.width < 768 && mobileMenu > 0) {
        console.log(`✅ Mobile menu found for ${viewport.name}`);
      } else if (viewport.width >= 768 && hiddenElements > 0) {
        console.log(`✅ Responsive hiding working for ${viewport.name}`);
      } else {
        console.log(`⚠️  Responsive behavior unclear for ${viewport.name}`);
      }

      // Test key interactions at this viewport
      const ctaButton = page.locator('button, a').filter({ hasText: /start|trial|sign/i }).first();
      if (await ctaButton.isVisible()) {
        const isClickable = await ctaButton.isEnabled();
        console.log(`${viewport.name} CTA button: ${isClickable ? '✅ Clickable' : '❌ Disabled'}`);
      }
    }
  });
});
