/**
 * Authentication Flow E2E Tests
 * Tests user signup, login, email verification, and password reset
 */

import { test, expect, Page } from '@playwright/test';

test.describe('Authentication Flow', () => {
  let testUser = {
    email: `test-${Date.now()}@example.com`,
    password: 'TestPassword123!',
    firstName: 'Test',
    lastName: 'User'
  };

  test.beforeEach(async ({ page }) => {
    // Clear any existing auth state
    await page.context().clearCookies();
    await page.context().clearPermissions();
  });

  test('Complete user signup and login flow', async ({ page }) => {
    console.log('🧪 Testing: User Signup Flow');

    // Navigate to signup page
    await page.goto('/signup');
    console.log('📄 Navigated to signup page');

    // Fill out signup form
    await page.fill('input[name="firstName"]', testUser.firstName);
    await page.fill('input[name="lastName"]', testUser.lastName);
    await page.fill('input[name="email"]', testUser.email);
    await page.fill('input[name="password"]', testUser.password);
    await page.fill('input[name="confirmPassword"]', testUser.password);

    // Select FREE plan
    await page.click('input[value="FREE"]');

    console.log('📝 Filled signup form with:', testUser);

    // Submit signup form
    await page.click('button[type="submit"]');
    console.log('🚀 Submitted signup form');

    // Wait for API response
    await page.waitForResponse(response =>
      response.url().includes('/api/auth/signup') && response.status() === 201
    );
    console.log('✅ Signup API call successful');

    // Check for success message or email verification prompt
    const signupSuccess = await page.locator('text=Check your email').isVisible() ||
                          await page.locator('text=verification').isVisible();

    if (signupSuccess) {
      console.log('✅ Signup successful - email verification required');
    } else {
      console.log('⚠️  Signup may have failed - checking for error messages');
      const errorText = await page.locator('.text-red-500, .text-red-600').first().textContent();
      if (errorText) {
        console.log('❌ Signup error:', errorText);
      }
    }

    // Try to login immediately (should fail without verification)
    await page.goto('/login');
    await page.fill('input[name="email"]', testUser.email);
    await page.fill('input[name="password"]', testUser.password);
    await page.click('button[type="submit"]');

    // Should show email verification required error
    const verificationError = await page.locator('text=email not verified, text=verification required').isVisible();
    if (verificationError) {
      console.log('✅ Email verification correctly required for login');
    } else {
      console.log('⚠️  Login behavior unexpected - may need email verification setup');
    }
  });

  test('Login with existing user', async ({ page }) => {
    console.log('🧪 Testing: User Login Flow');

    // Use a pre-existing test user or skip if none exists
    const existingUser = {
      email: process.env.TEST_USER_EMAIL || 'admin@example.com',
      password: process.env.TEST_USER_PASSWORD || 'password123'
    };

    await page.goto('/login');
    console.log('📄 Navigated to login page');

    // Fill login form
    await page.fill('input[name="email"]', existingUser.email);
    await page.fill('input[name="password"]', existingUser.password);
    console.log('📝 Filled login form');

    // Submit login
    await page.click('button[type="submit"]');
    console.log('🚀 Submitted login form');

    // Check for success or error
    await page.waitForTimeout(2000); // Wait for redirect or error

    const currentUrl = page.url();
    if (currentUrl.includes('/dashboard') || currentUrl.includes('/welcome')) {
      console.log('✅ Login successful - redirected to dashboard');
    } else if (await page.locator('text=Invalid email or password').isVisible()) {
      console.log('⚠️  Login failed - invalid credentials (expected for test user)');
    } else {
      console.log('⚠️  Login result unclear - checking page content');
      const pageTitle = await page.title();
      console.log('📄 Current page title:', pageTitle);
    }
  });

  test('Password reset flow', async ({ page }) => {
    console.log('🧪 Testing: Password Reset Flow');

    await page.goto('/login');

    // Click forgot password link
    const forgotPasswordLink = page.locator('text=Forgot password?, text=forgot, text=reset').first();
    if (await forgotPasswordLink.isVisible()) {
      await forgotPasswordLink.click();
      console.log('🔗 Clicked forgot password link');

      // Fill reset form
      await page.fill('input[name="email"]', testUser.email);
      await page.click('button[type="submit"]');
      console.log('📧 Requested password reset for:', testUser.email);

      // Check for success message
      const resetMessage = await page.locator('text=reset instructions sent, text=check your email').isVisible();
      if (resetMessage) {
        console.log('✅ Password reset request successful');
      } else {
        console.log('⚠️  Password reset flow may need implementation');
      }
    } else {
      console.log('⚠️  Forgot password link not found');
    }
  });

  test('Admin access control', async ({ page }) => {
    console.log('🧪 Testing: Admin Access Control');

    // Try to access admin routes without authentication
    await page.goto('/admin');
    console.log('📄 Attempted to access admin dashboard');

    const currentUrl = page.url();
    if (currentUrl.includes('/login') || currentUrl.includes('/signup')) {
      console.log('✅ Admin access correctly blocked - redirected to auth');
    } else {
      console.log('⚠️  Admin access may not be properly protected');
      const pageContent = await page.textContent('body');
      console.log('📄 Admin page content preview:', pageContent?.substring(0, 200));
    }
  });

  test('Navigation and UI elements', async ({ page }) => {
    console.log('🧪 Testing: Navigation and UI Elements');

    await page.goto('/');
    console.log('📄 Loaded homepage');

    // Check for key navigation elements
    const navElements = [
      'Strategy Playground',
      'Chat with AI',
      'Pricing',
      'Login',
      'Sign Up'
    ];

    for (const element of navElements) {
      const isVisible = await page.locator(`text=${element}`).isVisible();
      console.log(`🔍 "${element}": ${isVisible ? '✅ Visible' : '❌ Not found'}`);
    }

    // Check for pricing CTA
    const pricingCTA = await page.locator('text=Start.*Trial, text=Start.*trial').first();
    if (await pricingCTA.isVisible()) {
      const ctaText = await pricingCTA.textContent();
      console.log('💰 Main CTA found:', ctaText);
    } else {
      console.log('⚠️  Main pricing CTA not found');
    }
  });
});
