# Database Setup Guide

This guide will help you set up the MySQL database for the Backtesting AI Platform.

## Prerequisites

- MySQL Server (8.0 or later recommended)
- Node.js 18+ (for Prisma CLI)

## Quick Setup

### 1. Install MySQL

**Ubuntu/Debian:**
```bash
sudo apt update
sudo apt install mysql-server
sudo mysql_secure_installation
```

**macOS (with Homebrew):**
```bash
brew install mysql
brew services start mysql
mysql_secure_installation
```

**Windows:**
Download from [MySQL official website](https://dev.mysql.com/downloads/mysql/)

### 2. Create Database and User

Log into MySQL as root:
```bash
mysql -u root -p
```

Create database and user:
```sql
-- Create database
CREATE DATABASE backtest_db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

-- Create user (replace 'your_password' with a strong password)
CREATE USER 'backtest_user'@'localhost' IDENTIFIED BY 'your_password';

-- Grant privileges
GRANT ALL PRIVILEGES ON backtest_db.* TO 'backtest_user'@'localhost';

-- Flush privileges
FLUSH PRIVILEGES;

-- Exit MySQL
EXIT;
```

### 3. Configure Environment Variables

Create a `.env` file in the project root:

```bash
# Copy the example file
cp .env.example .env
```

Update the `DATABASE_URL` in `.env`:
```bash
DATABASE_URL="mysql://backtest_user:your_password@localhost:3306/backtest_db"
```

**Important:** Replace `your_password` with the actual password you set for the database user.

### 4. Initialize Database

Run the automated setup script:
```bash
npm run db:setup
```

This will:
- ✅ Test database connection
- ✅ Create all tables and relationships
- ✅ Seed initial data (strategy templates)
- ✅ Generate Prisma client

## Manual Setup (Alternative)

If you prefer manual control:

```bash
# Generate Prisma client
npm run db:generate

# Push schema to database
npm run db:push
```

## Database Schema Overview

### Core Tables

- **`users`** - User accounts, authentication, subscriptions
- **`strategies`** - Trading strategies, templates, social features
- **`conversations`** - Chat conversations with AI
- **`messages`** - Individual chat messages

### Analytics Tables

- **`user_analytics`** - User behavior tracking
- **`strategy_analytics`** - Strategy performance metrics
- **`system_analytics`** - System performance monitoring

### Social Features

- **`strategy_comments`** - User comments on strategies
- **`strategy_ratings`** - Star ratings for strategies
- **`strategy_shares`** - Forks, saves, bookmarks

### Support & Communication

- **`feedback`** - User feedback and support tickets
- **`notifications`** - In-app notifications

## Using Prisma Studio

View and edit your data with Prisma Studio:
```bash
npm run db:studio
```

This opens a web interface at `http://localhost:5555`

## Database Operations

### Common Queries

```typescript
import { DatabaseService } from '@/lib/database';

// Create a user
const user = await DatabaseService.createUser({
  email: 'user@example.com',
  password: 'securepassword',
  subscriptionTier: 'PRO'
});

// Find strategies by user
const strategies = await DatabaseService.findStrategiesByUser(userId);

// Record analytics
await DatabaseService.recordUserEvent(userId, 'STRATEGY_CREATED', {
  strategyId: 'strategy_123',
  domain: 'STOCKS'
});
```

### Direct Prisma Access

For complex queries, use the Prisma client directly:

```typescript
import { prisma } from '@/lib/database';

// Complex query example
const popularStrategies = await prisma.strategy.findMany({
  where: {
    isPublic: true,
    likes: { gte: 10 }
  },
  include: {
    user: { select: { firstName: true, lastName: true } },
    ratings: { select: { rating: true } }
  },
  orderBy: { likes: 'desc' },
  take: 20
});
```

## Backup & Recovery

### Backup Database
```bash
mysqldump -u backtest_user -p backtest_db > backup.sql
```

### Restore Database
```bash
mysql -u backtest_user -p backtest_db < backup.sql
```

## Production Deployment

### Environment Variables for Production

```bash
DATABASE_URL="mysql://user:password@production-host:3306/backtest_db"
NODE_ENV="production"
```

### Connection Pooling

Prisma automatically handles connection pooling. For high-traffic applications, consider:

```typescript
// In production, you might want to adjust connection limits
const prisma = new PrismaClient({
  log: ['error'],
  datasources: {
    db: {
      url: process.env.DATABASE_URL,
    },
  },
});
```

## Troubleshooting

### Connection Issues

1. **Check MySQL service is running:**
   ```bash
   sudo systemctl status mysql  # Linux
   brew services list | grep mysql  # macOS
   ```

2. **Verify credentials:**
   ```bash
   mysql -u backtest_user -p backtest_db
   ```

3. **Check DATABASE_URL format:**
   - Must include `mysql://`
   - Username and password required
   - Database name must exist

### Migration Issues

If schema changes cause issues:
```bash
# Reset database (⚠️ destroys data)
npm run db:push -- --force-reset

# Or manually drop and recreate
mysql -u root -p -e "DROP DATABASE backtest_db; CREATE DATABASE backtest_db;"
npm run db:setup
```

### Performance Issues

1. **Add database indexes** for frequently queried columns
2. **Enable query logging** in development
3. **Monitor slow queries** with MySQL's slow query log

## Next Steps

After database setup, you can:

1. **Start the application:** `npm run dev`
2. **View data in Prisma Studio:** `npm run db:studio`
3. **Implement authentication APIs** using the DatabaseService
4. **Add user registration and login** endpoints

The database foundation is now ready for the full application! 🎉
