#!/usr/bin/env python3
"""
Demo script showing walk forward analysis preventing overfitting
"""

import json
import sys
import os

# Add the project root to Python path
sys.path.insert(0, os.path.dirname(__file__))

from universal_backtesting import UniversalBacktestingEngine

def demo_walk_forward_prevents_overfitting():
    """Demonstrate how walk forward analysis prevents overfitting"""

    print("🎯 WALK FORWARD ANALYSIS - PREVENTING OVERFITTING")
    print("=" * 60)
    print()
    print("❓ PROBLEM: Curve-fitting occurs when strategies perform great on historical")
    print("   data but fail miserably on future unseen data.")
    print()
    print("✅ SOLUTION: Walk Forward Analysis tests strategies on future data segments")
    print("   to ensure robustness and prevent overfitting.")
    print()
    print("📊 DEMO: XAUUSD Trend Following Strategy with Walk Forward Validation")
    print()

    engine = UniversalBacktestingEngine()

    # Configuration for walk forward analysis
    walk_forward_config = {
        'in_sample_months': 1,     # 1 month for training/optimization
        'out_of_sample_months': 1, # 1 month for testing/validation
        'step_months': 1,          # Move forward 1 month each period
        'start_date': '2025-08-01',
        'end_date': '2025-11-21'
    }

    print("🎛️  Analysis Configuration:")
    print(f"   • In-sample period: {walk_forward_config['in_sample_months']} months (training)")
    print(f"   • Out-of-sample period: {walk_forward_config['out_of_sample_months']} months (testing)")
    print(f"   • Step size: {walk_forward_config['step_months']} months")
    print(f"   • Data range: {walk_forward_config['start_date']} to {walk_forward_config['end_date']}")
    print()

    # Run walk forward analysis
    result = engine.run_backtest(
        market="forex",
        strategy_name="trend_following",
        parameters={"pair": "XAUUSD", "stake": 100},
        timeframe="1hour",
        min_trades=5,
        walk_forward_periods=walk_forward_config
    )

    if not result.get('success') or 'walk_forward_analysis' not in result:
        print(f"❌ Analysis failed: {result.get('error', 'Unknown error')}")
        return

    wf = result['walk_forward_analysis']

    print("📈 WALK FORWARD RESULTS:")
    print("-" * 40)

    # Overall performance
    if wf.get('overall_statistics'):
        overall = wf['overall_statistics']
        print("Overall Performance Across All Test Periods:")
        print(".1f")
        print(".2f")
        print(f"   • Total P&L: ${overall['total_pnl']}")
        print(".1f")
        print(f"   • Best Period: ${overall['best_period_pnl']}")
        print(f"   • Worst Period: ${overall['worst_period_pnl']}")
        print(".2f")
        print()

    # Overfitting analysis
    if wf.get('overfitting_summary'):
        print("🎯 Overfitting Risk Assessment:")
        print("-" * 30)
        # Extract key parts from the summary
        summary_lines = wf['overfitting_summary'].split('\n')
        for line in summary_lines[:5]:  # Show first 5 lines
            if line.strip():
                print(f"   {line}")
        print()

    # Period-by-period breakdown
    if wf.get('periods') and len(wf['periods']) > 0:
        print("📋 Period-by-Period Validation:")
        print("-" * 35)

        periods = wf['periods']
        for i, period in enumerate(periods, 1):
            print(f"\nPeriod {i}:")

            in_sample = period['in_sample_period']
            out_sample = period['out_of_sample_period']
            analysis = period['overfitting_analysis']

            print(f"   Training ({in_sample['start_date']} to {in_sample['end_date']}):")
            print(f"     • {in_sample['metrics']['total_trades']} trades, {in_sample['metrics']['win_rate']}% win rate")
            print(f"     • P&L: ${in_sample['metrics']['total_pnl']}")

            print(f"   Testing ({out_sample['start_date']} to {out_sample['end_date']}):")
            print(f"     • {out_sample['metrics']['total_trades']} trades, {out_sample['metrics']['win_rate']}% win rate")
            print(f"     • P&L: ${out_sample['metrics']['total_pnl']}")

            risk_level = analysis['overfitting_risk_level']
            score = analysis['overfitting_score']
            print(f"   📊 Overfitting Risk: {risk_level} (Score: {score:.1f})")

            # Show performance decay
            if analysis.get('performance_decay_percent'):
                decay = analysis['performance_decay_percent']
                print(".1f")

    print()
    print("🎯 WALK FORWARD ANALYSIS BENEFITS:")
    print("-" * 40)
    print("✅ Prevents overfitting by testing on unseen future data")
    print("✅ Validates strategy robustness across different market conditions")
    print("✅ Provides realistic performance expectations")
    print("✅ Identifies parameter sensitivity and stability")
    print("✅ Gives confidence intervals for expected returns")
    print("✅ Professional-grade validation methodology")
    print()

    # Recommendations
    if wf.get('recommendations') and len(wf['recommendations']) > 0:
        print("💡 RECOMMENDATIONS:")
        print("-" * 20)
        for rec in wf['recommendations'][:3]:  # Show top 3
            print(f"• {rec}")
        print()

    print("🚀 CONCLUSION:")
    print("Walk forward analysis provides institutional-grade validation,")
    print("ensuring your strategies will perform in live markets, not just")
    print("on historical backtests. This prevents the most common trading mistake:")
    print("overfitting to past data that never repeats in the future.")
    print()
    print("🎪 Your AI trading platform now includes professional walk forward")
    print("   analysis - the gold standard for strategy validation!")

if __name__ == "__main__":
    demo_walk_forward_prevents_overfitting()
