#!/usr/bin/env python3
"""
Demo script showing parameter optimization capabilities
"""

import json
import sys
import os

# Add the project root to Python path
sys.path.insert(0, os.path.dirname(__file__))

from parameter_optimizer import ParameterOptimizer

def demo_parameter_optimization():
    """Demonstrate parameter optimization for trading strategies"""

    print("🎯 PARAMETER OPTIMIZATION DEMO")
    print("=" * 50)
    print()
    print("❓ PROBLEM: Finding the best parameters for trading strategies manually")
    print("   takes forever and often leads to overfitting!")
    print()
    print("✅ SOLUTION: Automated parameter optimization using:")
    print("   • Grid Search - Systematic testing")
    print("   • Random Search - Efficient sampling")
    print("   • Genetic Algorithms - Advanced evolution")
    print()
    print("💰 SUBSCRIPTION TIERS:")
    print("   • FREE: No optimization")
    print("   • PRO ($20/mo): Up to 20 evaluations")
    print("   • ELITE ($200/mo): Full optimization with all methods")
    print()

    optimizer = ParameterOptimizer()

    # Demo 1: Grid Search on Trend Following
    print("📊 DEMO 1: GRID SEARCH OPTIMIZATION")
    print("-" * 40)

    # Define parameter ranges for trend following
    parameter_ranges = {
        'short_ma_period': [5, 10, 15],        # Test 3 different short MA periods
        'long_ma_period': [20, 30, 40],        # Test 3 different long MA periods
        'stop_loss_pct': [0.02, 0.03, 0.04]   # Test 3 different stop loss levels
    }

    optimization_config = {
        'method': 'grid_search',
        'max_evaluations': 10,  # 3x3x3 = 27 combinations, but limit to 10
        'target_metric': 'sharpe_ratio'
    }

    print(f"Optimizing trend following strategy...")
    print(f"Parameters to optimize: {list(parameter_ranges.keys())}")
    print(f"Total possible combinations: 3×3×3 = 27")
    print(f"Method: {optimization_config['method']}")
    print(f"Max evaluations: {optimization_config['max_evaluations']}")
    print(f"Target metric: {optimization_config['target_metric']}")
    print()

    results = optimizer.optimize_strategy(
        market="forex",
        strategy_name="trend_following",
        base_parameters={"pair": "XAUUSD", "stake": 100},
        parameter_ranges=parameter_ranges,
        optimization_config=optimization_config,
        data_file=None,
        timeframe="1hour"
    )

    if results.get('success'):
        print("✅ Optimization completed successfully!")
        print()

        # Show results
        stats = results.get('statistics', {})
        print("📈 OPTIMIZATION RESULTS:")
        print(f"   • Total evaluations completed: {results['total_evaluations']}")
        print(f"   • Best {stats['target_metric'].replace('_', ' ').title()}: {stats['best_value']}")
        print(f"   • Average: {stats['average_value']}")
        print(f"   • Standard deviation: {stats['standard_deviation']}")
        print()

        # Show best parameters
        best = results.get('best_result', {})
        if best.get('parameters'):
            print("🏆 BEST PARAMETER SET:")
            for param, value in best['parameters'].items():
                print(f"   • {param.replace('_', ' ').title()}: {value}")
            print()

        # Show top 3 results
        top_3 = results.get('top_5_results', [])[:3]
        if top_3:
            print("🥇 TOP 3 PERFORMING COMBINATIONS:")
            for i, result in enumerate(top_3, 1):
                print(f"\n   Rank {i}:")
                for param, value in result['parameters'].items():
                    print(f"     • {param.replace('_', ' ').title()}: {value}")
                print(f"     • Sharpe Ratio: {result['target_metric']}")
                print(f"     • Win Rate: {result['key_metrics']['win_rate']}%")
                print(f"     • Total Profit: ${result['key_metrics']['total_profit']}")

    print()
    print("=" * 50)
    print()

    # Demo 2: Random Search
    print("📊 DEMO 2: RANDOM SEARCH OPTIMIZATION")
    print("-" * 40)

    # More parameters for random search
    parameter_ranges_random = {
        'short_ma_period': [5, 20],           # Range for random sampling
        'long_ma_period': [20, 50],
        'stop_loss_pct': [0.01, 0.05],
        'take_profit_pct': [0.02, 0.10]
    }

    optimization_config_random = {
        'method': 'random_search',
        'max_evaluations': 8,  # Random search with fewer evaluations
        'target_metric': 'total_profit'
    }

    print(f"Optimizing with random search...")
    print(f"Parameters: {list(parameter_ranges_random.keys())}")
    print(f"Method: {optimization_config_random['method']}")
    print(f"Evaluations: {optimization_config_random['max_evaluations']}")
    print(f"Target: {optimization_config_random['target_metric']}")
    print()

    results_random = optimizer.optimize_strategy(
        market="forex",
        strategy_name="trend_following",
        base_parameters={"pair": "XAUUSD", "stake": 100},
        parameter_ranges=parameter_ranges_random,
        optimization_config=optimization_config_random,
        data_file=None,
        timeframe="1hour"
    )

    if results_random.get('success'):
        best_random = results_random.get('best_result', {})
        if best_random.get('parameters'):
            print("🏆 RANDOM SEARCH BEST RESULT:")
            for param, value in best_random['parameters'].items():
                print(f"   • {param.replace('_', ' ').title()}: {value}")
            print(f"   • Total Profit: ${best_random['target_metric']}")
            print()

    print("🎯 PARAMETER OPTIMIZATION BENEFITS:")
    print("-" * 40)
    print("✅ Eliminates guesswork in parameter selection")
    print("✅ Prevents overfitting through systematic testing")
    print("✅ Finds optimal parameter combinations automatically")
    print("✅ Compares strategies on objective metrics")
    print("✅ Saves countless hours of manual testing")
    print("✅ Professional-grade optimization algorithms")
    print()
    print("💰 SUBSCRIPTION VALUE:")
    print("• PRO ($20/mo): Perfect for systematic traders")
    print("• ELITE ($200/mo): For serious strategy developers")
    print()
    print("🚀 Ready to optimize your trading strategies!")

if __name__ == "__main__":
    demo_parameter_optimization()
