#!/usr/bin/env python3
"""
Complete Advanced Backtesting Demo
Shows intrabar tick data + parameter optimization + walk forward analysis
"""

import json
import sys
import os

# Add the project root to Python path
sys.path.insert(0, os.path.dirname(__file__))

from universal_backtesting import UniversalBacktestingEngine
from parameter_optimizer import ParameterOptimizer

def demo_advanced_backtesting():
    """Demonstrate the complete advanced backtesting suite"""

    print("🚀 ADVANCED BACKTESTING SUITE DEMO")
    print("=" * 60)
    print()
    print("🎯 PROFESSIONAL TRADING FEATURES:")
    print("   ✅ Intrabar Tick Data - Access price movements within bars")
    print("   ✅ Parameter Optimization - Find optimal strategy settings")
    print("   ✅ Walk Forward Analysis - Prevent overfitting validation")
    print("   ✅ Subscription Tiers - Pro & Elite premium features")
    print()
    print("💰 SUBSCRIPTION TIERS:")
    print("   • FREE: Basic backtesting + walk forward")
    print("   • PRO ($20/mo): + Limited parameter optimization (20 evaluations)")
    print("   • ELITE ($200/mo): + Full optimization + intrabar data + genetic algorithms")
    print()

    engine = UniversalBacktestingEngine()

    # ==========================================
    # DEMO 1: Intrabar Tick Data Support
    # ==========================================
    print("📊 DEMO 1: INTRABAR TICK DATA ANALYSIS")
    print("-" * 50)

    print("❓ ADVANCED STRATEGIES NEED:")
    print("   • Price movements within each bar")
    print("   • Volume spikes during the bar")
    print("   • Intrabar volatility patterns")
    print("   • Tick-by-tick price action")
    print()

    # Test with intrabar ticks enabled
    result_with_ticks = engine.run_backtest(
        market="forex",
        strategy_name="trend_following",
        parameters={"pair": "XAUUSD", "stake": 100},
        timeframe="1hour",
        min_trades=5,
        use_intrabar_ticks=True
    )

    if result_with_ticks.get('success'):
        print("✅ Backtest with intrabar tick data completed!")
        verification = result_with_ticks.get('verification_data', {})
        print(f"   Data source: {verification.get('data_source', 'Unknown')}")
        print(f"   Total trades: {result_with_ticks.get('total_trades', 0)}")
        print()

        # Show that intrabar data is available
        if result_with_ticks.get('trades') and len(result_with_ticks['trades']) > 0:
            sample_trade = result_with_ticks['trades'][0]
            if 'intrabar_ticks' in str(sample_trade):
                print("📈 Intrabar tick data available for advanced analysis!")
                print("   Strategies can now access price movements within each bar")
            else:
                print("📊 Using standard OHLC data (intrabar ticks available on Elite tier)")
    print()

    # ==========================================
    # DEMO 2: Parameter Optimization (Elite Feature)
    # ==========================================
    print("🎯 DEMO 2: PARAMETER OPTIMIZATION (ELITE TIER)")
    print("-" * 50)

    optimizer = ParameterOptimizer()

    print("🔬 OPTIMIZING TREND FOLLOWING STRATEGY PARAMETERS")
    print()

    # Comprehensive parameter optimization
    parameter_ranges = {
        'short_ma_period': [3, 5, 8, 10, 12],    # More options for Elite users
        'long_ma_period': [15, 20, 25, 30, 40],  # Comprehensive testing
        'stop_loss_pct': [0.01, 0.02, 0.03, 0.04, 0.05],
        'take_profit_pct': [0.02, 0.05, 0.08, 0.10]
    }

    optimization_config = {
        'method': 'random_search',  # Elite users get all methods
        'max_evaluations': 15,       # Higher limits for Elite
        'target_metric': 'sharpe_ratio'
    }

    print(f"🎛️  Optimization Setup:")
    print(f"   • Method: {optimization_config['method']}")
    print(f"   • Evaluations: {optimization_config['max_evaluations']}")
    print(f"   • Parameters: {len(parameter_ranges)} ({', '.join(parameter_ranges.keys())})")
    print(f"   • Target: {optimization_config['target_metric'].replace('_', ' ').title()}")
    print()

    opt_results = optimizer.optimize_strategy(
        market="forex",
        strategy_name="trend_following",
        base_parameters={"pair": "XAUUSD", "stake": 100},
        parameter_ranges=parameter_ranges,
        optimization_config=optimization_config,
        timeframe="1hour"
    )

    if opt_results.get('success'):
        print("✅ Parameter optimization completed!")
        print()

        stats = opt_results.get('statistics', {})
        print("📊 OPTIMIZATION STATISTICS:")
        print(f"   • Evaluations: {opt_results['total_evaluations']}")
        print(f"   • Best Sharpe Ratio: {stats.get('best_value', 0):.3f}")
        print(f"   • Average: {stats.get('average_value', 0):.3f}")
        print(f"   • Std Dev: {stats.get('standard_deviation', 0):.3f}")
        print()

        # Show best parameters
        best = opt_results.get('best_result', {})
        if best.get('parameters'):
            print("🏆 OPTIMAL PARAMETERS FOUND:")
            for param, value in best['parameters'].items():
                print(f"   • {param.replace('_', ' ').title()}: {value}")
            print()

        # Show top performers
        top_3 = opt_results.get('top_5_results', [])[:3]
        if top_3:
            print("🥇 TOP 3 STRATEGY VARIATIONS:")
            for i, result in enumerate(top_3, 1):
                print(f"\n   #{i} Performance:")
                key_params = ['short_ma_period', 'long_ma_period', 'stop_loss_pct']
                for param in key_params:
                    if param in result['parameters']:
                        print(f"     • {param.replace('_', ' ').title()}: {result['parameters'][param]}")
                print(f"     • Sharpe Ratio: {result['target_metric']:.3f}")
                print(f"     • Win Rate: {result['key_metrics']['win_rate']}%")
    print()

    # ==========================================
    # DEMO 3: Walk Forward + Optimization Combo
    # ==========================================
    print("🎯 DEMO 3: WALK FORWARD + OPTIMIZATION COMBO")
    print("-" * 50)

    print("🔬 COMPLETE PROFESSIONAL WORKFLOW:")
    print("   1. Optimize parameters on historical data")
    print("   2. Validate with walk forward analysis")
    print("   3. Ensure no overfitting to past data")
    print()

    # Walk forward analysis on optimized parameters
    wf_config = {
        'in_sample_months': 2,      # Optimize on 2 months
        'out_of_sample_months': 1,  # Test on 1 month
        'step_months': 1,           # Move forward monthly
        'start_date': '2025-08-01',
        'end_date': '2025-11-21'
    }

    wf_result = engine.run_backtest(
        market="forex",
        strategy_name="trend_following",
        parameters={"pair": "XAUUSD", "stake": 100, "short_ma_period": 5, "long_ma_period": 20},
        timeframe="1hour",
        min_trades=5,
        walk_forward_periods=wf_config
    )

    if wf_result.get('success') and 'walk_forward_analysis' in wf_result:
        wf = wf_result['walk_forward_analysis']

        if wf.get('overall_statistics'):
            overall = wf['overall_statistics']
            print("🎯 WALK FORWARD VALIDATION:")
            print(f"   • Periods tested: {overall['total_periods']}")
            print(f"   • Average win rate: {overall['average_win_rate']}%")
            print(f"   • Average monthly P&L: ${overall['average_monthly_pnl']}")
            print(f"   • Profitable periods: {overall['profitable_periods_percent']}%")
            print()

        if wf.get('overfitting_summary'):
            print("📋 OVERFITTING ANALYSIS:")
            lines = wf['overfitting_summary'].split('\n')[:4]  # Show key lines
            for line in lines:
                if line.strip():
                    print(f"   {line}")
            print()

    print("=" * 60)
    print()

    # ==========================================
    # FEATURE COMPARISON
    # ==========================================
    print("💰 SUBSCRIPTION FEATURE COMPARISON:")
    print("-" * 50)

    features = {
        'FREE': [
            '✅ Basic backtesting (OHLC)',
            '✅ Detailed trade logs',
            '✅ Performance metrics',
            '✅ Walk forward analysis',
            '✅ Strategy visualization',
            '❌ Intrabar tick data',
            '❌ Parameter optimization',
            '❌ Advanced algorithms'
        ],
        'PRO ($20/mo)': [
            '✅ All FREE features',
            '✅ Limited optimization (20 eval)',
            '✅ Grid search method',
            '✅ Random search method',
            '❌ Intrabar tick data',
            '❌ Genetic algorithms',
            '❌ Unlimited evaluations'
        ],
        'ELITE ($200/mo)': [
            '✅ All PRO features',
            '✅ Intrabar tick data',
            '✅ Genetic algorithms',
            '✅ Unlimited evaluations',
            '✅ Advanced analytics',
            '✅ Priority support',
            '✅ Custom strategy development'
        ]
    }

    for tier, feature_list in features.items():
        print(f"🎯 {tier}:")
        for feature in feature_list:
            print(f"   {feature}")
        print()

    print("🚀 YOUR AI TRADING PLATFORM NOW OFFERS:")
    print("   ✅ Institutional-grade backtesting")
    print("   ✅ Advanced parameter optimization")
    print("   ✅ Intrabar tick data analysis")
    print("   ✅ Walk forward overfitting protection")
    print("   ✅ Professional subscription tiers")
    print("   ✅ Complete trading strategy suite")
    print()
    print("💰 Ready to take your trading to the next level!")

if __name__ == "__main__":
    demo_advanced_backtesting()
